	print	"MON32"

mon32	module

;32032 Debugger

op_bpt	equ	h'f2		;BPT opcode

	mode	sb

;Register names must be in same order as T_RG_NM
;Must be CPU (reg 0-7 first), MMU, FPU (reg 0-7 first)

reg_cpu:
m_r0:	.blkb	4
m_r1:	.blkb	4
m_r2:	.blkb	4
m_r3:	.blkb	4
m_r4:	.blkb	4
m_r5:	.blkb	4
m_r6:	.blkb	4
m_r7:	.blkb	4
m_psr:	.blkb	4
m_pc:	.blkb	4
m_sb:	.blkb	4
m_fp:	.blkb	4
m_usp:	.blkb	4
m_isp:	.blkb	4
m_intb:	.blkb	4
m_mod:	.blkb	4

reg_dbg:
	cond	cpu532
m_dcr:	 blkd
m_bpc:	 blkd
m_dsr:	 blkd
m_car:	 blkd
reg_cfg:
m_cfg:	 blkd
	cend

;MMU registers

reg_mmu:
	cond	cpu032 and w_mmu
m_bp0:	 blkd
m_bp1:	 blkd
m_pf0:	 blkd
m_pf1:	 blkd
m_sc:	 blkd
m_msr:	 blkd
m_bcnt:	 blkd
m_pt0:	 blkd
m_pt1:	 blkd
m_eia:	 blkd
	cend
	cond	cpu532
m_mcr:	 blkd
m_msr:	 blkd
m_tear:	 blkd
m_pt0:	 blkd
m_pt1:	 blkd
	cend

;FPU registers

reg_fpu:
m_f0:	blkb	4
m_f1:	blkb	4
m_f2:	blkb	4
m_f3:	blkb	4
m_f4:	blkb	4
m_f5:	blkb	4
m_f6:	blkb	4
m_f7:	blkb	4
m_l1:	blkd
m_l0:	blkd
m_l3:	blkd
m_l2:	blkd
m_l5:	blkd
m_l4:	blkd
m_l7:	blkd
m_l6:	blkd
m_fsr:	blkb	4
reg_last:

m_tfpu:	blkd			;Type of FPU installed

;Hardware configuration:
;Bit 0 = Interrupt controller present
;Bit 1 = FPU present
;Bit 2 = MMU present
;Bit 3 = Custom configuration present
	cond	cpu032
m_cfg:	 blkd			;Current configuration
	cend
m_dintb: blkd			;Debugger's internal INTBASE

m_tmpc:	.blkb	4		;Temporary PC storage for interpret command
m_retadr: blkb	4		;Return address storage for BRK_CHK

m_win1:	ds	1		;Window 1 command input if not 0
m_windx: ds	1		;Current window index
m_windat: blkd	win_max

mprompt: blkb			;Prompt character

;Bit 7 set if M_TMPC holds real PC value
;Bit 6 set if skipping listing of next opcode
;Bit 5 set if not displaying "Break" message
;Bit 4 set if clearing trace bits and RETT at trace trap
;Bit 3 set if returning from condition test
;Bit 2 set if T_TRC holds address of BPT to install at trace trap
;Bit 1 set if tracing only non-sequential instructions
;Bit 0 set if current instruction is always traced

m_trpbrk:
	.blkb	1
m_trcop: blkb	1		;Opcode storage for break condition test

t_trc: blkb	4		;Address of breakpoint to be traced past
t_brk:	blkb	2*4		;T_PAS must immediately follow T_BRK
t_pas:	blkb	16*4

{Additional storage for breakpoint addresses:
1 byte original opcode
4 byte counter, 0=breakpoint disabled
4 byte address to call for conditional break, 0=off
buf_siz bytes of terminated text, optional command

break when count=1 or condition test returns <>0 value on stack

1 block is for trace; count, condition active
2 blocks for temporary breakpoints, count always 1, condition always off
16 blocks for permanent breakpoints; count, condition active
}

tbrk_siz equ	1+4+4+buf_siz	;Size of break blocks

t_trcop: blkb	tbrk_siz
t_brkop: blkb	2*tbrk_siz
t_pasop: blkb	16*tbrk_siz

mbrk_num: blkb	1		;Number of current breakpoint
m_trcmd: blkb	1		;Current trace mode S/N

;Trace buffer
trc_siz	equ	1024		;Number of Dwords to store, power of 2 only
trc_buf: blkb	4*trc_siz	;Actual buffer for trace addresses
trc_cnt: blkb	4		;Number of addresses traced

cmd_buf: blkb	2		;Last command buffer
lin_abt: blkb			;Line abort flag (quit if <> 0)

m_dcount: blkb	4		;Default count for dump
m_dlast: blkb	4		;Default next address for dump
m_dmpmd: blkb	1		;Default dump mode

m_lcount: blkb	4		;Default count for list
m_llast: blkb	4		;Default next address for list
m_lstmd: blkb	1		;Default dump list

mlk_beg: blkd			;Starting address for Look command
mlk_end: blkd			;Ending address for Look command
mlk_str: ds	21		;1 byte count, 20 bytes for search string

m_slast: blkb	4		;Last source load ending address

sym_last: ds	8		;Last M_SYMADR, M_SYMALF for S<y>mbol command

;Disassembler variables

m_amodp: .blkb	4		;Pointer to current mode base pointer
m_amode: .blkb	1		;Mode code
m_lstmod: blkb	4		;Current mode for listing
m_genm:	ds	1		;Gen 1 or gen 2 selection

old_fp:	.blkb	4		;Original FP storage
old_sp:	.blkb	4		;Original SP storage
old_sb:	.blkb	4		;Original SB storage
old_pc:	.blkb	4		;Original PC storage

siz_fp:	.blkb	4		;FP size (last value)
siz_sp:	.blkb	4		;SP size (last value)
siz_sb:	.blkb	4		;SB size (last value)
siz_pc:	.blkb	4		;PC size (last value)

m_disend:			;Last address to disassemble
	.blkb	4
m_hexptr:			;Pointer to hex string array
	.blkb	4
m_hexctr:			;Length of hex string
	.blkb	4
m_ascptr:			;Point to opcode string
	.blkb	4
m_ascctr:			;Length of opcode string
	.blkb	4

mac_ex:	ds	1		;Macro executing if NZ
m_opcod: blkb	2		;Storage for opcode (format/index)
b_ps	equ	8		;Pseudo op bit
b_mac	equ	9		;MACRO name if set

b_alist	equ	0		;Assembler listing flag
b_2pass	equ	1		;2 pass assembly for listing (instead of 1)
b_xrf	equ	2		;Cross reference enable (set = build XREF)
b_maclst equ	3		;List macro expansions if set
b_fclst	equ	4		;List false conditional blocks if set
b_1lst	equ	5		;Pass 1 listing if set
m_dolst: .blkb	1		;List flag, 0=no listing
m_mdolst: blkb			;Outermost macro saves status here
m_width: ds	4		;Number of bytes of object code to display

m_mode:	.blkb	1

msb_mod: blkd			;Current module's SB

m_int:	.blkb	1
m_siz:	.blkb	1		;Immediate operand size
m_flt:	.blkb	1
m_cond:	.blkb	2
m_dspsz: .blkb	1		;Default displacement size (1,2,4)
m_und:	blkb	1		;Undefined parameter in line

gen1:	blkb	1+1+(2+8)+(2+8)
gen2:	blkb	1+1+(2+8)+(2+8)

;Condition stack:
;3 bytes PC_PTR, 1 byte(bits 0-3 pseudo-op, bits 4-7 condition)
cond_siz equ	4*200		;200 entries of 4 bytes each
cond_ptr: blkb	4		;Index to condition stack
cond_stk: blkb	cond_siz	;Storage for condition stack

m_cndcod: blkb	4		;Current condition code
m_endadr: blkb	4		;END conditional address
m_nd1adr: blkb	4		;END + 1 conditional address
m_qndadr: blkb	4		;QEND conditional address
m_elsadr: blkb	4		;ELSE conditional address

m_cndclr: blkb	4		;Last line where condition stack was clear

;MACROs
mac_stk: blkb	5		;1 byte opening pseudo-op, 4 byte address
mac_rd:	blkb	4		;Macro read level counter
mac_lbl: blkb	4		;Local label counter
mac_cnt: ds	1		;MACRO counter to use with link stack
m_macclr blkb	4		;Last line where no macros in progress

;1 byte gen, 1 byte index
;2 byte status, 4 byte value for 2 displacements

m_gen1:	.blkb	h'e
m_gen2:	.blkb	h'e

hb:	.blkb	h'80

var_ptr: blkb	4		;Pointer to next available slot in VAR_BUF
var_siz: equ	h'400		;Size of variable array
var_buf: blkb	var_siz

m_proc:	blkb	1		;Procedure status
m_prcstk: blkb	4		;Size of procedures local stack
m_prcreg: blkb	1		;Register bit map for procedure

b_comnt	equ	0		;Comment block (set = comment)
b_asmbl	equ	1		;Conditional assembly flag (set = don't)
b_xlat	equ	2		;Translation enable (set = translate)
b_tmpx	equ	3		;Temporary translation enable
b_fp	equ	4		;0 = local FP storage, 1 = general storage
b_lxlat	equ	5		;Local symbol translation only
b_nscgnx equ	6		;NSCGNX if set
m_ascond: blkb			;Assembler condition status
m_macond: blkb			;Saved here during outermost macro execution
acnd_lst: blkb	4		;Last line of terminated conditional assembly

acnd_stk: blkb	16		;Conditional assembly stack
acnd_ptr: blkb	1		;Pointer to conditional assembly stack

m_srcptr: blkb	4		;Pointer to untranslated source code
m_lincnt: blkb	4		;Line counter for source code
m_linlst: blkb	4		;Line counter for listing

m_symalf: blkd			;Current T_SYMALF index
t_symlcl: blkd			;Index to local symbols
m_sym1st: blkd			;Start of nested local symbols
m_disfp: blkb	4		;Disassembler pointer to FP base in symbols
lcod_fp	equ	8		;Symbol link code for FP base pointer
lcod_mac equ	80h		;Link code for macro block, ORed with count

	mode	pc
	disp	2		;Default displacement size

;Debugger starts executing here, all CPU/MMU/FPU registers initialized
;as necessary, must be in Supervisor mode

debugger:
	br	begin		;Begin monitor execution

;Check break count and condition, FS on return if break required
;R0 holds address of break count, condition; original R0 in M_R0
;Registers not saved yet, stack still as setup by trap
;R0 and stack restored on exit

brk_chk:
	cmpqd	2,(r0)		;Never let count go below 1
	if	hi		;Return NE if ready to break
	 movd	m_r0,r0
	 ret
	endif
	addqd	-1,(r0)		;Decrement count

;Count not yet expired
;Enter here if tracing and skipping count test/decrement

brk_chkt:
	cmpqd	0,4(r0)		;See if condition routine enabled
	if	eq		;No break if no condition check
	 movd	m_r0,r0
	 ret
	endif

;Condition check required, 4(R0) holds target address

	sbitb	3,m_trpbrk	;Next breakpoint is return from condition
	movd	tos,m_retadr	;Save this routine's return address
	movd	tos,m_pc	;Routine's return address
	movzwd	tos,m_mod
	movzwd	tos,m_psr
	tbitb	log flag_s,m_psr ;See which stack is in use
	if	fs
	 bispsrw flag_s		;Enable user stack for now
	endif
	movqd	0,tos		;Condition flag
	movd	m_pc,tos	;Condition's return address is same BPT
	bicpsrw	flag_s		;Force supervisor stack now
	movw	m_psr,tos
	movw	m_mod,tos
	movd	4(r0),tos	;Target address
	movd	m_r0,r0		;Restore original R0
	cond	cpu532
	 lprw	mod,4(sp)	;In case of DE mode
	 lprd	sb,m_sb
	cend
	rett			;Continue with condition test


;Breakpoint trap: PSR, MOD, PC on stack from bottom to top

trp_bpt:
	bicw	flag_p,6(sp)	;Clear trace pending flag
	cbitb	3,m_trpbrk	;Check for return from condition test
	if	fs		;Return from condition, 8(SP) holds result
	 movd	tos,m_pc	;Save PC
	 movzwd	tos,m_mod	;Save MOD
	 movzwd	tos,m_psr	;Save PSR
	 tbitb	log flag_s,m_psr ;See which stack was in use
	 if	fs
	  bispsrw flag_s	;Select user stack if needed
	 endif
	 cmpqd	0,tos		;Check and clear EQ/NE flag
	 bicpsrw flag_s		;Force supervisor stack again
	 movw	m_psr,tos
	 movw	m_mod,tos
	 movd	m_pc,tos	;Supervisor stack restored
	 jump	(m_retadr)
	endif

	movd	r0,m_r0		;BRK_CHK requires original R0 here
	save	[r1,r4]
	movd	8(sp),r4	;Address of break
	addr	t_brk-4+4*18,r1	;Address of last break
	movd	2+16,r0
	skpsd	b,u		;Locate break
	restore	[r1,r4]

	if	fs		;Break found if FS
	 addqd	-1,r0		;Make 0-17
	 muld	tbrk_siz,r0	;Offset to count, condition data
	 addr	t_brkop+1[r0:b],r0 ;Point to count
	 movb	-1(r0),m_trcop	;Store current opcode
	 bsr	brk_chk
	 if	eq		;Not breaking if EQ
	  movb	m_trcop,((sp))	;Restore original opcode
	  cond	cpu532
	   cinv	i,a,r0		;Opcode not written to instruction cache
	  cend
	  movd	(sp),t_trc	;Restore BPT to this address
	  sbitb	2,m_trpbrk	;Trace trap must restore BPT to (T_TRC)
	  tbitb	log flag_t,6(sp) ;See if trace was active
	  if	fc
	   sbitb 4,m_trpbrk	;Just run after one trace if FC
	  endif
	  orb	flag_t,6(sp)	;Set trace bit
	  cond	cpu532
	   lprw	mod,4(sp)	;In case of DE mode
	   lprd	sb,m_sb
	  cend
	  rett
	 endif
	endif

	bsr	sav_reg
	bsr	de_bpt		;Remove breakpoints

	bsr	find_brk
	if	cs
	 movb	r0,mbrk_num	;Save current break number if valid
	endif

	cbitb	5,m_trpbrk	;Don't display message if set
	if	fc
	 bsr	dsp_msg:d
	 byte	"Break trap ",0
	 movzbd	mbrk_num,r5
	 addqb	-2,r5		;Min break number for command
	 cmpb	15,r5
	 if	hs
	  bsr	dsp_r5:d
	 endif
	 bsr	do_cr:d
	endif

	movzbd	mbrk_num,r5
	cmpb	15,r5
	if	hs
	 muld	tbrk_siz,r5
	 addr	t_brkop+1+4+4[r5:b],r1 ;Address of command line
	 bsr	lin_cmd
	endif

	br	trp_bktc	;Continue with general trap routine

;Trace trap: PSR, MOD, PC on stack from bottom to top

trp_trc:
	cbitb	2,m_trpbrk	;Restoring BPT from (T_TRC) if set
	if	fs
	 movb	op_bpt,(t_trc)
	 cond	cpu532
	  cinv	i,a,r0		;BPT not written to instruction cache
	 cend
	endif

	cbitb	4,m_trpbrk	;Clear FLAG_T and RETT if set
	if	fs
	 bicw	flag_p+flag_t,6(sp)
	 cond	cpu532
	  lprw	mod,4(sp)	;In case of DE mode
	  lprd	sb,m_sb
	 cend
	 rett
	endif

;Check sequential trace options

	cbitb	0,m_trpbrk	;Always do this instruction if set
	if	fc		;Check sequential option if FC
	 tbitb	1,m_trpbrk	;Sequential opcodes only if set
	 if	fs
	  save	[r0]
	  extsd	(4(sp)),r0,0,4	;Next opcode to execute
	  cmpqb	b'0010,r0	;Format 1
	  if	eq
	   extsd (4(sp)),r0,4,4	;Op within format
	   cmpqb 3,r0		;BSR,RET,CXP,RXP
	   if	lo
	    restore [r0]
	    cond cpu532
	     lprw mod,4(sp)	;In case of DE mode
	     lprd sb,m_sb
	    cend
	    rett		;Sequential, just do it
	   endif
	  else
	   cmpb	b'1010,r0	;Format 0, always non-sequential
	   if	ne
	    andb 3,r0		;Just check bits 0,1
	    cmpqb 2,r0		;Check for non-integer 2 byte opcode
	    if	eq
	     restore [r0]
	     cond cpu532
	      lprw mod,4(sp)	;In case of DE mode
	      lprd sb,m_sb
	     cend
	     rett
	    else		;Two byte integer-type opcode
	     extsd (4(sp)),r0,2,2
	     cmpqb 3,r0
	     if	ne		;Format 4 if NE (sequential)
	      restore [r0]
	      cond cpu532
	       lprw mod,4(sp)	;In case of DE mode
	       lprd sb,m_sb
	      cend
	      rett
	     else
	      extsd (4(sp)),r0,4,3 ;This determines format 2 or 3
	      cmpqb 7,r0	;Format 3 if EQ
	      if eq
	       tbitb 8,(4(sp))
	       if fs
	        extsd 1(4(sp)),r0,0,3 ;Opcode within format
	        cmpqb 7,r0	;CASEi
	        if ne		;Sequential if NE
		 restore [r0]
	         cond cpu532
	          lprw mod,4(sp) ;In case of DE mode
	          lprd sb,m_sb
	         cend
		 rett
	        endif
	       endif		;Bit 8=0 is always non-sequential
	      else		;This is format 2
	       extsd (4(sp)),r0,4,3
	       cmpqb 4,r0	;ACBi
	       if	ne
	        restore [r0]
	        cond cpu532
	         lprw mod,4(sp) ;In case of DE mode
	         lprd sb,m_sb
	        cend
		rett
	       endif
	      endif
	     endif
	    endif
	   endif
	  endif
	  sbitb	0,m_trpbrk	;Do next one too
	  restore [r0]
	 endif
	endif

	movd	r0,m_r0		;BRK_CHK requires original R0 here

;Load up backtrace buffer now

	movd	trc_cnt,r0
	andd	trc_siz-1,r0	;Wrap around as needed
	movd	(sp),trc_buf[r0:d] ;Store current PC
	addqd	1,trc_cnt	;Increment counter

	movd	(sp),t_trc	;Return address saved here
	movb	(t_trc),t_trcop	;Save current opcode
	movb	op_bpt,(t_trc)	;Install BPT here for BRK_CHK
	cond	cpu532
	 cinv	i,a,r0		;BPT not written to instruction cache
	cend
	addr	t_trcop+1,r0	;Address of pointer to use
	bicw	flag_p+flag_t,6(sp) ;Trace bits off now
	bsr	brk_chk		;Check for breakpoint
	movb	t_trcop,(t_trc)	;Restore original opcode
	cond	cpu532
	 cinv	i,a,r0		;BPT not written to instruction cache
	cend
	if	eq		;No break yet if EQ
	 orb	flag_t,6(sp)	;Trace bit on again
	 cond cpu532
	  lprw mod,4(sp)	;In case of DE mode
	  lprd sb,m_sb
	 cend
	 rett
	end

	bsr	sav_reg
	bsr	de_bpt		;Remove any breakpoints
	cmpqd	2,t_trcop+1	;Display remaining count if not expired
	if	ls
	 bsr	dsp_msg:d
	 byte	"Remaining trace count is ",0
	 movd	t_trcop+1,r5
	 bsr	dsp_r5:d
	 bsr	do_cr:d
	end

	addr	1+4+4+t_trcop,r1 ;Execute command if any
	bsr	lin_cmd
	br	trp_bktc

;Remove any installed breakpoints, restore original opcodes

de_bpt:

;Remove all breakpoints and restore opcodes

	save	[r0,r1,r2,r3]
	movzbd	16+2,r0		;16 pass points, 2 break points
	addr	t_brk,r1
	addr	t_brkop,r2

	until	eq
	 movd	(r1),r3
	 cmpqd	0,1(r2)		;Break disabled if count is 0
	 if	ne
	  cmpb	op_bpt,(r3)	;See if BPT is installed
	  if	eq
	   movb (r2),(r3)	;Restore original opcode
	  endif
 	 endif
	 addqd	4,r1
	 addd	tbrk_siz,r2
	 addqd	-1,r0
	 cmpqd	0,r0
	endu

	cond	cpu532
	 cinv	a,i,r0		;Opcode not written to instruction cache
	cend

	restore	[r0,r1,r2,r3]
	ret


traps:	cond	cpu532
	 tbitb	cfg_de,m_cfg
	 if	fs
	  save	[all]
	  bsr	get_sb:d	;Get trapped SB
	  movd	r5,m_sb		;Sav_reg will use sim_fpu and change stored SB
	  restore [all]
	 endif
	cend
	bsr	sav_reg
	movqb	0,m_trpbrk	;Reset break/trace status bits
	bsr	de_bpt		;Remove any breakpoints

trp_bktc:			;Break and trace enters here
	cbitb	7,m_trpbrk	;Restore temp PC if bit 7 set in M_TRPBRK
	if	fs
	 movd	m_tmpc,m_pc
	endif

;List current instruction if bit 6 clear in M_TRPBRK

	cbitb	6,m_trpbrk
	if	fc
	 movd	m_pc,r4		;Default list address
	 bsr	find_brk
	 if	cs		;This is a break address if CS
	  cmpb	r0,mbrk_num	;If at this break then list M_PC
	  if	ne
	   muld	tbrk_siz,r0	;Offset to current T_BRKOP entry
	   cmpqd 0,1+t_brkop[r0:b] ;See if break enabled
	   if	ne
	    cmpqd 0,1+4+t_brkop[r0:b] ;See if condition test enabled
	    if	ne		;Enabled if NE
	     tbitb 3,m_trpbrk	;Condition test not begun if FC
	     if	fc
	      movd 1+4+t_brkop[r0:b],r4 ;List condition address instead
	     endif
	    endif
	   endif
	  endif
	 endif
	 bsr	dis_dsp:d
	 movd	r4,m_llast	;Load M_LLAST from current PC
	endif

	br	cmd_lp

;See if M_PC is a breakpoint
;FIND_BK1 uses value in R4
;Number returned in R0, CS if found, CC if not found
;FIND_BK2 also checks break count, returns CS if found and count<>0

find_bk1:
	cmpqd	0,r4		;Everything is this if off
	if	ne
	 save	[r1]
	 addr	t_brk-4+4*18,r1	;Address of last break
	 movd	2+16,r0
	 skpsd	b,u		;Locate break
	 restore [r1]
	 addqd	-1,r0
	else
	 movqd	-1,r0		;Nothing found
	 bicpsrb flag_c		;Clear carry
	endif
	ret

find_brk:
	save	[r4]
	movd	m_pc,r4
	bsr	find_bk1
	restore	[r4]
	ret

find_bk2:
	bsr	find_bk1
	if	cs		;Only check count if break found
	 save	[r0]
	 muld	tbrk_siz,r0
	 cmpqd	0,t_brkop+1[r0:b]
	 restore [r0]
	 if	eq
	  bicpsrb flag_c	;CC if count is 0
	 endif	
	endif
	ret


;Save registers, assumes return address on stack followed by:
;  PC, MOD, PSR from SVC/trap - these values are cleared on exit

sav_reg:
	movd	r0,m_r0		;Save all general registers
	movd	r1,m_r1
	movd	r2,m_r2
	movd	r3,m_r3
	movd	r4,m_r4
	movd	r5,m_r5
	movd	r6,m_r6
	movd	r7,m_r7

	sprd	sp,m_isp	;Address of SAV_REG return
	addd	3*4,m_isp	;Address of stack before BSR and SVC
	movd	4(sp),m_pc
	movzwd	4+4+2(sp),m_psr
	bicw	flag_p+flag_t,m_psr ;Clear trace bits

	bispsrw	flag_s		;Select user stack
	sprd	sp,m_usp
	bicpsrw	flag_s		;Back to supervisor stack

	sprd	fp,m_fp
	sprd	intbase,m_intb
	lprd	intbase,m_dintb	;Load internal table

	cond	cpu532
	 sprd	dcr,m_dcr
	 sprd	bpc,m_bpc
	 sprd	dsr,m_dsr
	 sprd	car,m_car
	 sprd	cfg,m_cfg
	cend

	movzwd	4+4(sp),m_mod	;MOD address
	cond	cpu532
	 tbitb	cfg_de,m_cfg
	 if	fc		;Not DE mode if fc
	cend
	  movd	(m_mod),m_sb	;Get SB now from MOD table
	cond	cpu532
	 endif
	cend

	bsr	sav_mf		;Save MMU, FPU as appropriate
	bispsrw	flag_i		;Interrupts must be on
	ret	4+2+2		;Remove PC, MOD, PSR from stack

sav_mf:
	tbitb	cfg_f,m_cfg	;See if FPU present
	if	fs
	 sfsr	m_fsr		;Must be 1st to keep trap type field
	 movl	f0,m_f0
	 movl	f2,m_f2
	 movl	f4,m_f4
	 movl	f6,m_f6
	 cmpqd	0,m_tfpu	;FPU type =0 if 32081
	 if	ne
	  movl	l1,m_l1
	  movl	l3,m_l3
	  movl	l5,m_l5
	  movl	l7,m_l7
	 endif
	else
	 cmpqd	-1,m_fsr
	orif	ne
	endif

	tbitb	cfg_m,m_cfg	;Check for MMU
	if	fs
	 cond	cpu032 and w_mmu
	  smr	msr,m_msr
;	  smr	eia,m_eia
	  smr	ivar1,m_eia	;532 equivalent
	  smr	bpr0,m_bp0
	  smr	bpr1,m_bp1
	  smr	pf0,m_pf0
	  smr	pf1,m_pf1
	  smr	sc,m_sc
;	  smr	bcnt,m_bcnt
	  smr	tear,m_bcnt	;532 equivalent
	  smr	ptb0,m_pt0
	  smr	ptb1,m_pt1
	 cend
	 cond	cpu532
	  smr	mcr,m_mcr
	  smr	msr,m_msr
	  smr	tear,m_tear
	  smr	ptb0,m_pt0
	  smr	ptb1,m_pt1
	 cend
	endif	
	ret


;Clear all breakpoints

brk_off:
	save	[r0,r1,r2]
	movzbd	2+16-1,r0
	addr	t_brk,r1
	addr	4(r1),r2
	movqd	0,0(r1)
	movsd

	addr	t_brkop,r1	;Break control table
	addr	tbrk_siz(r1),r2
	movqd	0,1(r1)		;Default count (off)
	movqd	0,1+4(r1)	;Default condition (off)
	movb	cr,1+4+4(r1)	;Default command is none
	movzwd	tbrk_siz*17,r0	;Count for move
	movsb
	restore	[r0,r1,r2]
	ret


;1st time initialization

begin:
	sprd	intbase,m_dintb	;Initialize internal table
	cond	cpu532
	 sprd	sb,m_sb		;In case of DE mode
	cend
	cond	cpu032
	 save	[all]
	 bsr	get_cfg:d	;Get configuration in R5
	 movd	r5,m_cfg	;Sav_reg needs this
	 restore [all]
	cend
	bsr	sav_reg		;Save all registers
	cond	not don
	 orw	flag_u+flag_s,m_psr ;Default to user mode/stack
	cend

	sprd	sb,r5
	addd	-4(r5),r5	;Next address after SB block
	addqd	-1,r5
	movd	r5,mem_siz	;Highest usable address
	bsr	get_fpu:d	;FPU type in R5
	cbitb	31,r5		;Ignore simulation bit
	subd	h'32081,r5	;0 if 32081, <>0 for anything else
	movd	r5,m_tfpu

	movzbd	dev_pr,r0
	movzbd	dev_smv,r1
	bsr	dev_rte:d	;Route printer output to system message device

	movb	dev_smv,tx_dev	;Output via system message device
	movqb	0,m_win1	;Don't use window 1 for command input is 0
	movqb	0,m_windx	;Start in window 0
	movzbd	win_max-1,r0
	addr	m_windat,r1
	addr	4(r1),r2
	movqd	-1,(r1)		;Invalidate all windows
	movsd
	bsr	get_win:d	;Current SMO window in R5
	movd	r5,m_windat

	movw	cr*256+" ",cmd_buf ;Default command terminated by CR
	movb	prompt,mprompt	;Current prompt character

	bsr	brk_off		;Fill break and pass points with 0
	addr	t_trcop,r1	;Trace control table
	movqd	1,1(r1)		;Default count
	movqd	0,1+4(r1)	;Default condition (off)
	movb	cr,1+4+4(r1)	;Default command is none
	movb	"S",m_trcmd	;Default mode is sequential
	movqd	0,trc_cnt	;Trace counter starts at 0
	movqb	0,m_trpbrk	;Trap/breakpoint status

	movb	"B",m_dmpmd	;Initialize default dump mode, byte
	movqd	0,m_dlast	;Next dump address
	movzbd	h'40,m_dcount	;Default dump count
	movb	"N",m_lstmd	;Default list mode, normal
	movqd	0,m_llast	;Next list address
	movzbd	8,m_lcount	;Default list count
	movqd	0,mlk_beg	;Starting search address
	movqd	0,mlk_end	;Ending search address
	movqb	0,mlk_str	;0 string length
	movd	mon_src,m_slast ;Next source load address
	movqd	0,r7		;Initial PC_OFST
	bsr	lnk_cst:d	;Multi-file assembler initialization
	bsr	asm_cst:d	;Single file assembler initialization
	movqb	0,m_dolst	;All list options off
	movw	exp b_sign+16,m_radix ;Hex, signed, Algebraic

;Display sign-on message

	bsr	clr_scr:d
	bsr	dsp_msg:d
	db	"Kotekan Debugger: "
	db	datemsg
	db	cr,lf
	db	"  Copyright 1991 by Digital Arts",cr,lf
	byte	'  Press "?" <RETURN> for command summary.',cr,lf
	byte	0

	addr	trp_trc,r4
	movzbd	9,r5		;Trace trap
	bsr	set_trpadr:d
	addr	trp_bpt,r4
	movzbd	8,r5		;Breakpoint trap
	bsr	set_trpadr:d
	addr	m_sb,r4
	movzbd	8,r5
	bsr	set_sbadr:d	;Store SB here if DE mode breakpoint
	addr	m_sb,r4
	movzbd	9,r5
	bsr	set_sbadr:d	;Store SB here if DE mode trace trap
	addr	traps,r4
	bsr	set_dbgadr:d	;New debugger entry point

cmd_lp:
	cond	cpu532
	 cinv	a,i,d,r0	;Just in case
	cend

	movb	"I",m_pass	;Immediate mode

	addr	var_buf,var_ptr	;Clear variable buffer
;	MOVD	M_LINK,M_LNKPTR	;Clear any temporary usage here

	movqd	0,1+t_brkop	;Cancel 1st temporary breakpoint
	movqd	0,1+tbrk_siz+t_brkop ;Cancel 2nd temporary breakpoint
	movqd	0,t_brk		;Cancel addresses too so
	movqd	0,t_brk+4	;FIND_BRK won't find these
	andb	b'1001,m_trpbrk	;Clear all but condition test and trace next

	proc
	reg	[]
buf1:	blkb	120		;Input buffer
buf2:	blkb	120		;Translation buffer
;BUF1:	BLKB	BUF_SIZ		;Input buffer
;BUF2:	BLKB	BUF_SIZ		;Translation buffer
	code

	addr	buf1,r1		;Input buffer
	addr	buf2,r2		;Translation buffer
	movd	r1,r3
	subd	r2,r3		;Max input length
	movqd	0,r0		;Current input length

	movb	m_ascond,tos
	bicb	exp b_xlat+exp b_tmpx+exp b_lxlat,m_ascond ;Translation off
	tbitb	0,m_win1	;Check for command entry in window 1
	if	fs
	 movb	m_windx,tos	;Current window selection
	 movqd	1,r5
	 bsr	set_wndw:d	;Switch in window 1
	 bsr	lin_inp:d	;Get command line in window 1
	 movzbd	tos,r5		;Original window
	 bsr	set_wndw:d	;Restore original window
	else
	 bsr	lin_inp:d	;No special command window
	endif
	movb	tos,m_ascond

	cmpb	cr,(r2)		;If null line, then repeat last command
	if	eq
	 addr	cmd_buf,r1
;	 CMPB	"G",(R1)	;Don't do this by default
;	 IF	NE
	  bsr	parse
;	 ENDIF
	else
	 movb	(r2),cmd_buf	;New current command
	 bsr	lin_cmd
	endif

	bsr	dsp_msg:d
	db	vc_scr:#,vcs_nln:#,0 ;CR if not in col. 0

	pend
	br	cmd_lp

;R1 => command line, do it

lin_cmd:
	lproc
	reg	[]
buf1:	blkb	120		;Translation buffer
;BUF1:	BLKB	BUF_SIZ		;Translation buffer
	code

	begin
	 movzbd (r1),r4
	 cmpb	"|",r4		;Check for multi-command line
	 if	eq
	  addqd 1,r1		;Skip this
	  cmpb	"|",(r1)	;See if " " or CR needed
	  if	eq
	   addqd 1,r1		;Skip this too
	   bsr	do_cr:d
	  else
	   movb " ",r5
	   bsr	char_tx:d
	  endif
	 endif
	 movzbd	(r1),r4
	 cmpb	trm_lin:#,(m_delim)[r4:b]
	while	lo
	 addr	buf1,r2
	 bsr	bld_lin:d
	 save	[r1]		;Pointer to line terminator
	 addr	buf1,r1
	 bsr	parse		;Got line, do what it says
	 restore [r1]		;Pointer to line terminator
	 cmpqb	0,lin_abt	;Check line abort flag, quit now if set
	quit	ne
	endw

	bsr	dsp_msg:d
	db	vc_scr,vcs_nln,0 ;CR if not in col. 0

	pend


;Error routine
error:	bsr	dsp_msg:d
	.byte	$"???",cr,lf,0
	bispsrb	flag_f
	ret	0

;Verify validity of float in 0(0(SP))
;Load into 2(R5) and set B_DEF as required
;(R5) holds type

vrf_flt:
	lproc
ptr:	blkd			;Pointer to float to install in R5
	reg	[r2,r3]
	code

	movd	ptr,r2		;Pointer to float
	extsb	1(r5),r3,b_size-8,4
	cmpqb	4,r3		;Check for float
	if	eq
	 cbitb	b_def,(r5)	;Make undefined
	 extsb	2(r2),r3,23-16,8 ;Exponent
	 cmpqb	-1,r3		;Exponent may not be all 1s
	 if	ne
	  cmpqb	0,r3		;Exp may not be 0 if mantissa is not 0
	  if	ne
	   movfl (r2),2(r5)
	   sbitb b_def,(r5)	;Valid now
	  else
	   extsd (r2),r3,0,23	;Mantissa
	   cmpqd 0,r3		;Mantissa must be 0 if exp is 0
	  orif eq
	  endif
	 endif
	else
	 cmpqb	5,r3		;64 bit long
	 if	eq
	  cbitb	b_def,(r5)	;Make undefined
	  extsw 6(r2),r3,52-48,11 ;Exponent
	  cmpw h'7ff,r3		;Exponent may not be all 1s
	  if	ne
	   cmpqw 0,r3		;Exp may not be 0 if mantissa is not 0
	   if	ne
	    movl (r2),2(r5)
	    sbitb b_def,(r5)	;Valid now
	   else
	    extsd 4(r2),r3,0,52-32 ;MSDword of mantissa
	    ord (r2),r3	;LSDword of mantissa
	    cmpqd 0,r3	;Mantissa must be 0 if exp is 0
	   orif eq
	   endif
	  endif
	 else
	  cmpqb	6,r3		;Variable size float
	 orif	eq
	 endif
	endif

	pend


;Expression evaluator, R1 => expression
;Like EX but evaluate using ES_GEN instead of ES
;Value returned in R5, F set on return if undefined

exp_gen:
	lproc
	reg	[r2]
buf:	blkb	2+10+10		;Need room for addressing mode
	code

	addr	buf,r5
	bsr	exp_gnsb

	ibitb	b_def,(r5)
	tbitb	b_def,(r5)	;FS if undefined
	if	fc
	 extsb	1(r5),r2,b_size-8,4
	 cmpqb	3,r2		;Error if not integer
	 if	lo
	  bispsrb flag_f
	 endif
	endif
	movd	2(r5),r5
	pend


;Expression evaluator, R1 => expression
;Like EX but evaluate using ES_GEN instead of ES
;Value returned in (R5)

exp_gnsb:
	lproc
	reg	[r0,r2,r3,r4,r6,r7]
mbas_ptr: blkb	4		;Math stack base pointer
op_stak: blkb	16		;Op stack for 16 math operators
	code

	movd	lmod,tos
	bsr	mod_prm		;Update register storage
	movd	math_ptr,mbas_ptr ;Base for evaluation

	tbitb	b_rpn,m_radix
	if	fs
	 bsr	exgn_rpn
	else
	 bsr	exgn_alg
	endif

	bsr	math_pop:d

	pend


exgn_rpn:
	bsr	next:d
	bfs	return:d

	bsr	ex_calc
	tbitb	b_def,(r5)
	if	fs
	 bsr	math_psh:d
	 br	exgn_rpn
	endif

;Check for termination character

	movzbd	(r1),r2
	cmpqb	trm_exp,(m_delim)[r2:b]
	bhs	return:d

;Check for math operator
	
	bsr	math_sym:d	;Return code + 1 in R0, R0=0 and FC if none
	bsr	do_math:d	;Operator code + 1 in R0
	cmpqd	0,r0
	bne	exgn_rpn	;That was a math operator if NE

	movd	r1,tos		;Save current pointer
	bsr	es_gen
	bsr	math_psh:d	;Push value onto stack
	movd	tos,r2
	cmpd	r1,r2		;If no length, then exit
	beq	return:d
	br	exgn_rpn


;Special addressing mode, et.al. routine for monitor calculator
;Value returned in (R5), B_DEF set as appropriate
;R0 altered

ex_calc:
	movqb	0,(r5)		;Assume undefined for now
	cmpb	":",(r1)
	if	eq
	 bsr	math_pop:d
	 br	ex_coln:d	;Return adjusted value in (R5)
	endif
	cmpb	"[",(r1)
	if	eq
	 bsr	es_gen
	 bsr	math_psh:d
	 movzbd mop_add+1:#,r0
	 bsr	do_math:d	;Add index to addressing mode
	 br	math_pop:d	;Return complete indexed value in R5
	endif
	cmpb	"(",(r1)	;Must be addressing mode
	if	eq
	 bsr	math_pop:d	;Undo any default addressing mode
	 extsb	b_mode/8(r5),r0,0,4 ;Addressing mode
	 cmpb	8,r0		;Check for FP
	 if	eq
	  subd	m_fp,2(r5)
	 else
	  cmpb	10,r0		;Check for SB
	  if	eq
	   subd	msb_mod,2(r5)
	  else
	   cmpb	9,r0		;Check for SP
	   if	eq
	    tbitb 9,m_psr
	    if	fc
	     subd m_isp,2(r5)	;Try system stack first
	    else
	     subd m_usp,2(r5) ;User stack
	    endif
	   else
	    cmpb 11,r0
	    if	eq
	     subd m_pcofst,2(r5)
	    endif
	   endif
	  endif
	 endif
	 movb	h'24,b_mode(r5)	;Just make constant now
	 bsr	math_psh:d
	 bsr	rnfpspsb
	 tbitb	b_def,(r5)
	 if	fs
	  bsr	math_psh:d
	  movzbd mop_add+1:#,r0
	  bsr	do_math:d	;Add index to addressing mode
	  br	math_pop:d	;Return complete value with previous offset
	 else			;Must be memory relative
	  addqd 1,r1
	  bsr	next:d
	  bsr	exp_gnsb	;Get disp 1, adjusted for addressing mode
	  tbitb b_def,(r5)
	  if	fs
	   cmpb	")",(r1)	;Check for implied addressing
	   if	eq
;	    EXTSB B_MODE/8(R5),R0,0,4 ;Addressing mode
;	    CMPB 8,R0		;Check for FP
;	    IF	EQ
;	     ADDD M_FP,2(R5)
;	    ELSE
;	     CMPB 10,R0		;Check for SB
;	     IF	EQ
;	      ADDD MSB_MOD,2(R5)
;	     ELSE
;	      CMPB 9,R0		;Check for SP
;	      IF EQ
;	       TBITB 9,M_PSR
;	       IF FC
;	        ADDD M_ISP,2(R5) ;Try system stack first
;	       ELSE
;	        ADDD M_USP,2(R5) ;User stack
;	       ENDIF
;	      ENDIF
;	     ENDIF
;	    ENDIF
	   else
	    extsb b_mode/8(r5),r0,b_mode-8,4 ;Symbol type
	    cmpqb 4,r0		;Already adjusted if not constant
	    if	eq
;	     MOVB H'24,B_MODE/8(R5) ;This is constant
	     bsr math_psh:d
	     bsr rnfpspsb	;Get base value
	     tbitb b_def,(r5)
	     if	fs		;May not be there if recursive call
	      bsr math_psh:d
	      movzbd mop_add+1:#,r0
	      bsr do_math:d	;Pointer now on stack
	     endif
	     bsr math_pop:d
	    else
	     bsr math_psh:d
	     bsr rnfpspsb	;Skip if this
	     bsr math_pop:d	;Previously adjusted value
	    endif
	   endif
	   tbitb b_def,(r5)
	   if	fs
	    movd 2(r5),r0
	    movd (r0),2(r5)	;Replace pointer with contents
	    bsr math_psh:d
	    movzbd mop_add+1:#,r0
	    bsr do_math:d	;Add disp 2 now
	    bsr	math_pop:d
	    bsr	next:d
	   endif
	  endif
	  cmpb ")",(r1)
	  if	eq
	   addqd 1,r1
	  else
	   movqb 0,(r5)	;Invalid expression
	  endif
	 endif
	endif
	ret


;Expression evaluator for algebraic notation

exgn_alg:
	addr	op_stak,r6	;Base of op stack

exgnalg1:
	movqd	0,r3		;1st operator code
	movqd	0,r4		;2nd operator code
	movqd	0,r7		;Op stack index

;Check for 1 parameter operator

exgnalg2:
	bsr	next:d
	bfs	exgnxit
	movd	r1,r2		;Save original pointer
	bsr	math_sym:d
	cmpb	mth_1prm+1:#,r0	;See if 1 parameter operator
	ble	exgnalg5
	movd	r2,r1		;Restore original R1

;Check for opening parenthesis

	cmpb	"(",(r1)
	if	eq
	 movb	r3,r6[r7:b]	;Store on op stack
	 addqd	1,r7
	 movd	r7,2(r5)	;Op stack index
	 movw	exp b_def+h'2400:#,(r5) ;Defined constant
	 bsr	math_psh:d
	 movd	r6,2(r5)	;Current op stack base pointer
	 bsr	math_psh:d	;Save this too
	 addr	r6[r7:b],r6	;New base pointer
	 addqd	1,r1
	 br	exgnalg1
	endif

	movd	r1,tos		;Save current pointer
	bsr	es_gen
	bsr	math_psh:d	;Push value onto stack
	movd	tos,r2
	cmpd	r1,r2		;If no length, then exit
	beq	exgnxit

;Get next operand in R4 now

exgnalg3a:
	bsr	next:d
	bfs	exgnxit

;Check for closing parenthesis

	cmpb	")",(r1)
	if	eq
	 bsr	exgnxit		;Make sure this line is finished

	 movd	math_ptr,r2
	 subd	mbas_ptr,r2
	 cmpd	3*6,r2		;Need op stack data + result of ()
	 bgt	exgnxit		;No corresponding "(" if GT

	 bsr	stk_swp:d	;R6 to top of stack
	 bsr	math_pop:d
	 movd	2(r5),r6	;Old op stack base pointer
	 bsr	stk_swp:d
	 bsr	math_pop:d
	 movd	2(r5),r7	;Old op stack index

	 addqd	-1,r7
	 movzbd	r6[r7:b],r3
	 movqd	0,r4
	 addqd	1,r1
	 br	exgnalg3a
	endif

	bsr	ex_calc
	tbitb	b_def,(r5)
	if	fs
	 bsr	math_psh:d
	 br	exgnalg3a
	endif

	bsr	math_sym:d
	cmpqd	0,r0
	beq	exgnxit		;End of expression if no more operators
exgnalg5:
	movd	r0,r4

;2 operators in R3 and R4, if R3 is highest priority just do that one
; unless both are 1 parameter operators, then evaluate right to left

exgnalg5a:
	cmpb	t_opordr-1:d[r3:b],t_opordr-1:d[r4:b]
	blo	exgnalg6
	cmpb	mth_1prm+1,r4
	bls	exgnalg6	;1 parameter ops evaluate right to left
	movd	r3,r0
	bsr	do_math:d

	cmpqd	0,r7		;See if op stack is empty
	beq	exgnalg7
	addqd	-1,r7
	movzbd	r6[r7:b],r3
	br	exgnalg5a
exgnalg6:
	cmpqd	0,r3
	if	ne
	 movb	r3,r6[r7:b]
	 addqd	1,r7
	end
exgnalg7:
	movd	r4,r3
	movqd	0,r4
	br	exgnalg2

;Closing routine for EXGN_ALG
;Resolve any remaining operators/operands

exgnxit:
	cmpqd	0,r3		;See if any operator left in R3
	if	ne
	 movd	r3,r0
	 bsr	do_math:d
	 movqd	0,r3
	endif
	cmpqd	0,r4
	if	ne
	 movd	r4,r0
	 bsr	do_math:d
	 movqd	0,r4
	endif

	begin
	 cmpqd	0,r7		;Check for more operators
	while	ne
	 addqd	-1,r7
	 movzbd	r6[r7:b],r0
	 bsr	do_math:d
	endw
	ret


;Evaluate addressing mode or register expression
;Return result in 2(R5)

es_gen:
	save	[r0,r2,r3,r4,r6]
	bsr	es_gnsb
	restore	[r0,r2,r3,r4,r6]
	ret

es_gnsb:
	movqb	0,(r5)		;Undefined for now
	movd	r1,r3		;Save text pointer
	bsr	pad_blnk:d	;Get name in R4
	save	[r1]
	movzbd	(reg_last-reg_cpu)/4,r0 ;Total number of registers to R0
	addr	t_rg_nm-4[r0:d],r1 ;Point to last entry
	skpsd	u,b		;FS if match found
	restore	[r1]

	if	fs		;Got a match if FS
	 movw	h'ff lsh b_mode,(r5) 	;Undefined for now
	 addqd	-1,r0
	 cmpb	(reg_fpu-reg_cpu)/4,r0 ;F0
	 if	hi
	  movd	m_r0[r0:d],2(r5) ;Load register value
	  movw	exp b_def+(h'24 lsh b_mode),(r5) ;Defined constant
	 else
	  cmpb	(m_fsr-reg_cpu)/4,r0 ;FSR
	 orif	eq
	  subb	(m_f0-reg_cpu)/4,r0 ;Convert 0 offset from M_F0
	  cmpb	(m_f7-m_f0)/4,r0 ;L0-L7
	  if	lo		;L0-L7 if LO
	   tbitb 0,r0		;L1/L3/L5/L7 if even
	   if	fs
	    subb (m_l0-m_f0)/4,r0 ;F0/F2/F4/F6
	    movw h'54 lsh b_mode,(r5) ;This is a long
	   else
	    cmpqd 0,m_tfpu	;Can't be L1/L3/L5/L7 if 32081
	    if	ne
	     movw h'54 lsh b_mode,(r5) ;This is a long
	    endif
	   endif
	  else			;This is F0-F7
	   cmpw	":L",(r1)	;Check for long
	   if	ne
	    movw h'44 lsh b_mode,(r5) ;This is a float
	   else
	    tbitb 0,r0		;:L must be even register
	    if	fc
	     movw h'54 lsh b_mode,(r5) ;This is a long
	     addqd 2,r1
	    endif
	   endif
	  endif
	  addr	m_f0[r0:d],tos	;Address of value
	  bsr	vrf_flt
	 endif
	 ret
	endif

	movd	r3,r1		;Restore original text pointer
	cmpb	"(",(r1)
	beq	rnfpspsb	;Must be R0-R7, FP, SP, SB
	cmpb	"[",(r1)		
	if	eq		;Must be scaled index if so
	 cmpb	"R",1(r1)
	 bne	return:d
	 cmpb	":",3(r1)
	 bne	return:d
	 cmpb	"]",5(r1)
	 bne	return:d
	 movzbd	2(r1),r0	;Register number
	 subb	"0",r0
	 cmpqb	7,r0		;Must be 0-7
	 blo	return:d
	 movd	m_r0[r0:d],2(r5) ;Index value to scale
	 movb	4(r1),r6	;Scale factor
	 movqd	1,r4		;Scale value for byte
	 cmpb	"B",r6		;Byte scale
	 if	eq
	  muld	r4,2(r5)
	  addqd	6,r1
	  movw	exp b_def+(h'24 lsh b_mode),(r5) ;Defined constant
	 else
	  movqd	2,r4		;Word scale
	  cmpb	"W",r6
	 orif	eq
	  movqd	4,r4		;Dword scale
	  cmpb	"D",r6
	 orif	eq
	  movb	8,r4		;Qword scale
	  cmpb	"Q",r6
	 orif	eq
	 endif
	 ret
	endif

	movd	(r1),r2
	andd	h'ffffff,r2
	cmpd	"TOS",r2
	if	eq
	 addqd	3,r1
	 tbitb	9,m_psr
	 if	fc
	  movd	m_isp,2(r5)	;Try system stack first
	 else
	  movd	m_usp,2(r5)	;User stack
	 endif
	 movw	exp b_def+(h'24 lsh b_mode),(r5) ;Defined constant
	 ret
	else
	 cmpd	"EXT",r2
	 if	eq
	  addqd	3,r1
	  bsr	next:d
	  cmpb	"(",(r1)
	  bne	return:d
	  addqd	1,r1
	  bsr	ex:d		;Search current module only
	  tbitb	b_def,(r5)
	  bfc	return:d
	  bsr	next:d
	  cmpb	")",(r1)
	  bne	return:d
	  addqd	1,r1		;Disp 1 now in 2(R5)
	  movd	2(r5),r3	;Base index
	  movd	4(m_mod),r4	;Link base
	  movd	r4[r3:d],2(r5)	;Base address
	  movw	exp b_def+(h'26 lsh b_mode),(r5) ;Defined external
	  ret
	 endif
	endif

	cmpb	"@",(r1)
	if	eq
	 addqd	1,r1
	 bsr	next:d
	 bsr	es_mod
	 movb	h'25,B_MODE/8(R5) ;Absolute, don't alter b_def
	 ret
	endif

	bsr	es_mod
	tbitb	b_def,(r5)
	bfc	return:d
	extsb	1(r5),r2,b_mode-8,4 ;Implied addressing mode

	cmpb	8,r2		;FP memory space
	if	eq
	 addd	m_fp,2(r5)
	 ret
	endif

	cmpb	9,r2		;SP memory space
	if	eq
	 tbitb	9,m_psr
	 if	fc
	  addd	m_isp,2(r5)	;Try system stack first
	 else
	  addd	m_usp,2(r5)	;User stack
	 endif
	 ret
	endif

	cmpb	h'a,r2		;SB memory space
	if	eq
	 addd	msb_mod,2(r5)
	 ret
	endif

	cmpb	h'b,r2		;PC memory space
	if	eq
	 addd	m_pcofst,2(r5)
	endif
	ret

;Evaluate value in (R1), check all modules

es_mod:
	save	[r0,r2,r3]
	movd	sidx_end,m_symadr ;No label found
	movd	r1,r3		;Save starting value
	bsr	es:d
	tbitb	b_def,(r5)
	if	fc		;Try all modules for symbol match
	 movd	r3,r1
	 bsr	length:d
	 movqd	0,msym_beg
	 movd	tsym_end,msym_end
	 begin
	  bsr	sym2val:d
	 quit	ne		;Nothing at all if NE
	  movmw	sidx_of(r2),(r5),5
	  tbitb	b_def,(r5)
	 while	fc
	  addr	sidx_of+1(r2),msym_beg
	  subd	t_symbl,msym_beg ;Convert to offset
	 endw
	 addd	r0,r1
	endif
	tbitb	b_def,(r5)
	if	fs
	 movd	m_symadr,r2
	 cmpd	sidx_end,r2	;See if a symbol was tried
	 if	hi
	  movd	r2,tos
	  addd	t_symbl,tos
	  bsr	lbl_mod		;Convert label address to module data
	 endif
	endif
	restore	[r0,r2,r3]
	ret

;TOS holds label address, set up module parameters

lbl_mod:
	lproc
adr:	blkd
	reg	[]
	code

	movd	adr,tos
	bsr	set_mod:d	;Get module number
	bsr	mod_prm		;Set up parameters

	pend

;TOS holds PC address, set up module parameters

pc_mod:
	lproc
adr:	blkd
	reg	[]
	code

	movd	adr,tos
	bsr	adr_mod:d	;Get module number
	bsr	mod_prm		;Set up parameters

	pend


;TOS holds SB address, set up module parameters

sb_mod:
	lproc
adr:	blkd
	reg	[]
	code

	movd	adr,tos
	bsr	sb2mod:d	;Get module number
	bsr	mod_prm		;Set up parameters

	pend


;TOS holds module number, set up parameters

mod_prm:
	lproc
tmod:	blkd
	reg	[r0,r1]
	code

	movd	tmod,r0
	movd	mod_indx,r1
	cmpd	r0,r1
	if	hs
	 negd	r1,r0		;Set carry if R1<>0
	 subcd	r0,r0		;If R1=0 then R0=0, else R0=-1
	 addd	r1,r0		;R0=0 or R0=R1-1
	endif
	movd	r0,lmod
	movd	tsym_end,msym_end ;End of symbols
	movd	m_sb,msb_mod	;Module SB
	cmpqd	0,r1
	if	eq
	 movqd	0,msym_beg
	 movqd	0,m_pcofst
	else
	 movd	r0,r1		;Save mod number
	 muld	mod_tsiz,r0
	 addr	mod_tabl[r0:b],r0
	 movd	4+4+2+2*sidx_of(r0),m_pcofst ;PC physical code address
	 cmpqd	1,mod_indx	;If only 1 then use M_SB
	 if	lo
	  tbitb	2,lnk_prm	;Check for ONEMOD
	  if	fc
	   movd	4+4+2+2*sidx_of+4+4(r0),msb_mod ;Module SB
	  else
	   movd	mod_tabl+4+4+2+2*sidx_of+4+4,msb_mod ;1st module's SB
	  endif
	 endif
	 extsd	4+4+2(r0),msym_beg,0,sidx_bit
	 subd	mod_indx,r1
	 cmpqd	-1,r1
	 if	ne		;If not last then fetch end from next module
	  extsd mod_tsiz+4+4+2(r0),msym_end,0,sidx_bit
	  addd	t_symbl,msym_end
	 endif
	endif

	pend


;Check for (R0-R7), (FP), (SP), (SB)
;Return appropriate value in (R5)
;(R1) => "(" on entry
;R1 advanced if OK

rnfpspsb:
	movqb	0,(r5)
	cmpb	")",3(r1)
	bne	return:d
	cmpw	"FP",1(r1)
	if	eq
	 movd	m_fp,2(r5)
	 movw	exp b_def+(h'28 lsh b_mode),(r5) ;Defined FP
	 addqd	4,r1
	else
	 cmpw	"SP",1(r1)
	 if	eq
	  tbitb	9,m_psr
	  if	fc
	   movd	m_isp,2(r5)	;Try system stack first
	  else
	   movd	m_usp,2(r5)	;User stack
	  endif
	  movw	exp b_def+(h'29 lsh b_mode),(r5) ;Defined SP
	  addqd	4,r1
	 else
	  cmpw	"SB",1(r1)
	  if	eq
	   movd	msb_mod,2(r5)
	   movw	exp b_def+(h'2a lsh b_mode),(r5) ;Defined SB
	   addqd 4,r1
	  else
	   cmpb	"R",1(r1)
	   if	eq
	    save [r0]
	    movzbd 2(r1),r0	;Register number
	    subb "0",r0
	    cmpqb 7,r0		;Must be 0-7
	    if	hs
	     movd m_r0[r0:d],2(r5) ;Value
	     movw exp b_def+(h'24 lsh b_mode),(r5) ;Defined constant
	     addqd 4,r1
	    endif
	    restore [r0]
	   endif
	  endif
	 endif
	endif
	ret

;Line is in R1, do what it wants

parse:	movqb	0,lin_abt	;Clear abort flag
	bsr	next:d
	bfs	return:d
	movb	(r1),r4
	save	[r1]
	addr	end_set-1,r1	;Address of last entry
	movzbd	end_set-cmd_set,r0
	skpsb	u,b
	restore	[r1]
	bfc	error		;Not found it FC
	addqd	-1,r0		;Make first entry 0
	addqd	1,r1		;Point to parameters
ps_bas:	casew	jmp_tbl[r0:w]

cmd_set:	
	db	"?ABCDEFGHIJKLMOPQRSTUVWXYZ"
end_set:

jmp_tbl:
	word	do_menu-ps_bas	;Display menu
	word	do_asm-ps_bas	;Assemble
	word	do_pas-ps_bas	;Breakpoint
	word	do_clc-ps_bas	;Calculate
	word	do_dmp-ps_bas	;Dump
	word	do_edt-ps_bas	;Edit
	word	do_fil-ps_bas	;Fill
	word	do_go-ps_bas	;Go
	word	do_hex-ps_bas	;Hex file transmit
	word	do_intpt-ps_bas	;Interpret assembler input
	dw	do_join-ps_bas	;Join device routing
	dw	do_kotkn-ps_bas	;Exit to system
	word	do_lst-ps_bas	;List
	word	do_mov-ps_bas	;Move
	word	do_o-ps_bas	;Look for value in memory
	word	do_prl-ps_bas	;Parallel port ASCII file load
	dw	do_quik-ps_bas	;Quick setup
	word	do_reg-ps_bas	;Register display
	dw	sys_set-ps_bas	;System setup
	word	do_trc-ps_bas	;Trace
	word	do_unt-ps_bas	;Untrace (backtrace)
	word	do_val-ps_bas	;Validate memory
	word	do_wndw-ps_bas	;Select window
	dw	get_paus-ps_bas	;Wait for key, abort or continue
	dw	do_symbl-ps_bas	;Symbol table
	word	clr_scr-ps_bas	;Clear screen


dev_in:
	lproc
	reg	[]
buf:	ds	2+8
	code

	movqd	0,r0		;Length of name if string in (R5)
	movd	r0,r5		;No value to start
	movd	r0,r6		;No device
	bsr	next:d
	if	fc
	 movb	dev_smv+1,r6
	 cmpd	"SER1",(r1)
	 if	eq
	  addqd	4,r1
	 else
 	  movb	dev_pmv+1,r6
	  cmpd	"SER2",(r1)
	 orif	eq
	  movb	dev_pr+1,r6
	  cmpd	"PRTR",(r1)
	 orif	eq
	  addr	buf,r5
	  bsr	exp_gnsb
	  ibitb	b_def,(r5)	;Set if undefined
	  tbitb	b_def,(r5)	;FS if undefined
	  if	fc
	   extsb 1(r5),r6,b_size-8,4
	   cmpqb 3,r6		;Check for integer
	   if	hs
	    movd 2(r5),r5
	    movqd 0,r6		;No device
	   else			;Check for string (file name)
	    bispsrb flag_f	;In case of error
	    cond op_sys
	     cmpb 8,r6
	     if	eq
	      movzwd 2(r5),r0	;Length of name
	      movd 2+2(r5),r5	;Address of name
	      movb dev_m1+1,r6
	      bicpsrb flag_f	;OK
	     endif
	    cend
	   endif
	  endif
	 endif
	endif
	if	fs
	 bsr	error
	 movqd	0,r6
	endif

	pend

dev_out:
	bsr	dev_in
	bfs	return:d	;Error if FS
	cmpqb	0,r6		;No device code if 0
	beq	return:d
	addqb	-1,r6		;Backup to output device
;	cmpb	dev_pmk,r6
;	if	eq
;	 addqb	-2,r6		;Separate output device
;	else
;	 cmpb	dev_smk,r6
;	orif	eq
;	endif
	ret


;Get input device code in R6 (0=none),
;and value of expression in R5 (if not device)
;Device opened and F set on return if error

get_devic:
	save	[r0]
	bsr	dev_in
	bsr	get_dv1:b
	restore	[r0]
	ret

;Open device if any
;If 'A' or 'B' appended to device, set binary or ASCII file type

get_dv1:
	bfs	return:d
	cmpqd	0,r6
	beq	return:d
	cmpb	'A',(r1)
	if	eq
	 bsr	dev_binoff:d
	 addqd	1,r1
	else
	 cmpb	'B',(r1)
	 if	eq
	  bsr	dev_binon:d
	  addqd	1,r1
	 endif
	endif
	save	[r1]
	movd	r5,r1		;File name if any
	bsr	dev_open:d
	restore	[r1]
	ret

;Get output device code in R6 (0=none)
;and value of expression in R5 (if not device)
;Device opened and F set on return if error

devic_out:
	save	[r0]
	bsr	dev_out
	bsr	get_dv1
	restore	[r0]
	ret


;Display mod # on stack

dsp_mod:
	lproc
tmod:	blkd			;Mod number to display
	reg	[r0,r1,r2,r3,r4,r5]
buf:	ds	50		;Display buffer
	code

	movw	m_radix,tos
	movd	lmod,tos
	movw	16,m_radix	;Always unsigned hex here

	movd	tmod,r4		;Module to do
	movd	r4,tos
	bsr	mod_prm		;Set up parameters

	movd	r4,tos
	bsr	mod_name	;Display name

	muld	mod_tsiz,r4
	addr	mod_tabl[r4:b],r4
	bsr	dsp_msg:d	;Display PC now
	db	vc_scr,vcs_tab,35 ;Move to column 35
	db	$"PC: ",0
	movd	4+4+2+2*sidx_of+4(r4),r5 ;PC address
	movd	r5,tos
	bsr	dsp_aloc
	cmpqd	-1,r5
	if	ne
	 cmpqd	0,(r4)		;Check for 0 size
	 if	ne
	  save	[r5]
	  movb	"-",r5
	  bsr	char_tx:d
	  restore [r5]
	  addd	(r4),r5		;Add size
	  addqd	-1,r5		;Last address
	  bsr	dsp_r5:d
	 endif
	 bsr	dsp_msg:d
	 db	$" (",0
	 movd	(r4),r5		;Size
	 bsr	dsp_r5:d
	 movb	")",r5
	 bsr	char_tx:d
	endif
	bsr	do_cr:d

	bsr	dsp_msg:d	;Display SB now
	db	"  SB: ",0
	movd	4+4+2+2*sidx_of+4+4(r4),r5 ;SB address
	movd	r5,tos
	bsr	dsp_aloc
	cmpqd	-1,r5
	if	ne
	 cmpqd	0,4(r4)		;Check for 0 size
	 if	ne
	  save	[r5]
	  movb	"-",r5
	  bsr	char_tx:d
	  restore [r5]
	  addd	4(r4),r5	;Add size
	  addqd	-1,r5		;Last address
	  bsr	dsp_r5:d
	 endif
	 bsr	dsp_msg:d
	 db	$" (",0
	 movd	4(r4),r5	;Size
	 bsr	dsp_r5:d
	 movb	")",r5
	 bsr	char_tx:d
	endif

	bsr	dsp_msg:d	;Display MOD
	db	vc_scr,vcs_tab,35 ;Move to column 35
	db	"MOD: ",0
	movd	4+4+2+2*sidx_of+4+4+4(r4),tos
	bsr	dsp_aloc
	bsr	do_cr:d

	movqd	0,r0		;Link entry counter
	addr	4+4(r4),r1	;Link entry pointer => 1st offset
	movd	4+4+2+2*sidx_of+4+4+4+4(r4),r2 ;Address of link table
	begin
	 movzwd	(r1),r3
	 cmpqw	-1,r3
	while	ne
	 bsr	dsp_msg:d
	 db	"  Link ",0
	 movd	r0,r5
	 bsr	dsp_r5:d
	 bsr	dsp_msg:d
	 db	$" @",0
	 movd	r2,tos
	 bsr	dsp_aloc	;Absolute address
	 
	 addd	t_cxp,r3	;Absolute pointer to current value
	 extsd	2(r3),r4,0,13	;MOD number
	 bsr	dsp_msg:d
	 db	$": ",0
	 movd	r4,tos
	 bsr	mod_name	;Display name and number
	 bsr	dsp_msg:d
	 db	$", ",0
	 
	 save	[r0,r1]
	 extsd	3(r3),r6,5,3	;Type of this MOD entry
	 movd	4(r3),r5	;Actual value of symbol
	 bsr	dsp_r5:d
	 cmpqb	b'011,r6	;PC (3), SB (2): set bit 3
	 if	hs
	  sbitb	3,r6
	 endif			;B_MODE now in R4
	 movb	":",r5
	 bsr	char_tx:d
	 addr	t_mode:d[r6:d],r1 ;Mode label
	 movqd	4,r0
	 bsr	trl_blnk:d
	 bsr	str_out:d
	 movqd	0,r0
	 addr	buf,r1		;Display from here
	 movd	4(r3),r5	;Actual value of symbol
	 movd	lmod,tos	;Save current setting
	 movd	r4,tos		;Target module number
	 bsr	mod_prm
	 cmpqb	b'110,r6	;Check for external type
	 if	eq
	  movb	b'1011,r6	;Treat as PC relative for symbol search
	 endif
	 cmpb	b'1011,r6	;Check for PC relative
	 if	eq
	  addd	m_pcofst,r5	;Subtracted by SYM_R5
	 endif
	 bsr	sym_r5:d	;Build symbol name in R1/R0
	 bsr	str_out:d	;Output string
	 bsr	do_cr:d
	 movd	tos,r1		;Original LMOD
	 movd	r1,tos
	 bsr	mod_prm		;Restore original
	 restore [r0,r1]
	 
	 movd	r3,r1		;Current pointer => last
	 addqd	1,r0		;Advance counter
	 cmpqd	-1,r2		;Don't change if undefined
	 if	ne
	  addqd	4,r2		;Advance address
	 endif
	endw

	movd	tos,r2
	movd	r2,tos
	bsr	mod_prm		;Restore original MOD
	movw	tos,m_radix	;Restore original setting
	pend

;Display "Unallocated" if -1, else display value on stack

dsp_aloc:
	lproc
reg5:	blkd
	reg	[r5]
	code

	movd	reg5,r5
	cmpqd	-1,r5
	if	eq
	 bsr	dsp_msg:d
	 db	"Unallocated",0
	else
	 bsr	dsp_r5:d	;Starting address
	endif
	pend

mod_name:
	lproc
num:	blkd
	reg	[r0,r1,r4,r5]
	code

	movd	num,r4
	muld	mod_tsiz,r4
	addr	mod_tabl[r4:b],r4 ;Pointer to proper table
	bsr	dsp_msg:d
	db	"Module ",0
	movd	num,r5
	bsr	dsp_r5:d
	extsd	4+4+2+sidx_of(r4),r5,0,sidx_bit ;Offset to name
	cmpd	sidx_end,r5	;See if name defined
	if	ne
	 addd	t_symbl,r5
	 extsd	(r5),r0,0,5	;Length of name
	 bsr	lbl_nam:d	;Get pointer to name in R1
	 movb	"<",r5
	 bsr	char_tx:d
	 bsr	str_out:d	;Display name
	 movb	">",r5
	 bsr	char_tx:d
	endif

	pend


;Advance to next string, ignore commas
;F set on exit if nothing
;F clear if good text follows
;  Z set if comma was passed

comma::	bsr	next:d
	bicpsrb	flag_z		;NE if error
	bfs	return:d

;Return NE if character is not a comma

	cmpb	",",0(r1)
	bne	return:d

;This is a comma, advance to next character and return EQ

	addqd	1,r1
	bsr	next:d
	bicpsrb	flag_z		;NE if error
	bfs	return:d

;Got a keeper, set Z flag

	bispsrb	flag_z
	ret	0


;Get line and echo characters until CR
;Input buffer in R1, translation buffer in R2, max size in R3
;Number of characters returned in R0

lin_inp::
	save	[r5]
	extsd	m_radix,r5,0,5	;Current radix selection
	cmpb	16,r5
	if	eq
	 movb	"H",r5		;Hex
	else
	 addb	"0",r5
	 cmpb	"9",r5
	 if	lo
	  addqb 7,r5
	 endif
	endif
	bsr	char_tx:d	;Display radix before prompt

	movb	mprompt,r5
	bsr	char_tx:d
	restore	[r5]

;Get a line of input from keyboard, echo and return in (R1), length in R0
;Translate line when input is complete
;R3 holds max number of characters to receive

lin_inp2::
	movzwd	r3,r0		;Max count to bits 0-15 of R0
	bsr	lin_ins:d
	save	[r0,r1,r2]	;Number of characters
	bsr	bld_lin:d	;Translate R1 into R2
	bsr	do_cr:d
	restore	[r0,r1,r2]	;Number of characters returned
	ret

;End of MON32
