	print	"FPU32"

	disp	1

;FPU software simulation

;These are the FPU pseudo-registers

	mode	sb
freg_f0: blkd
freg_f1: blkd
freg_f2: blkd
freg_f3: blkd
freg_f4: blkd
freg_f5: blkd
freg_f6: blkd
freg_f7: blkd
freg_l1: blkd	2
freg_l3: blkd	2
freg_l5: blkd	2
freg_l7: blkd	2
freg_fsr: blkd

	mode	pc

;------------------------
; Intialize fpu simulator
;------------------------

simfpu_init:
	save	[r0,r1,r2,r3,r4,r5]
	addr	freg_f0,r1
	addr	4(r1),r2
	movzbd	2*8,r0
	movqd	0,(r1)
	movsd			;Fill all registers with 0

	cond	cpu532		;Same SB storage for UND and FPU traps
	 addr	10*tde_max+tde_ssb(tde_vct),3*tde_max+tde_psb(tde_vct)
	cend

	movzbd	10,r5
	addr	fpu_sbr:w,r4
	sprd	psr,r3
	sprd	mod,r2
	bsr	instal_trp	;Intercept with FPU simulator
	restore	[r0,r1,r2,r3,r4,r5]
	ret
	

;;Find left most one in SRC operand
;;Return bit number in DEST operand
;;FS if SRC>1, DEST=0 if not

lmo	macro	src,dest
	local	adr1,adr2
	movzbd	31,dest
adr1:	tbitb	dest,src
	bfs	adr2:b
	acbb	-1,dest,adr1
adr2:	mend


;;Negate Quad word

negq	macro	lsd,msd
	negd	lsd,lsd
	addcd	0,msd
	negd	msd,msd
	mend


;Main subroutine
;Update PC, SP, PSR status bits
;Destination RMW operands are not written if a trap is requested
;Complete 32081 operand/32381 register simulation except:
;  Rounding is toward 0 (truncation) unless round to nearest (mode 0) selected

fpu_sbr:
	tbitb	log flag_t,4+2(sp) ;Check for trace
	if	fs
	 orw	flag_p,4+2(sp)	;Set trace pending too
	endif
	movd	r0,tos
	tbitb	log flag_s,4+4+2(sp) ;Check stack selection
	if	fs		;User stack if FS
	 bispsrw flag_s		;Select user stack
	 sprd	sp,r0		;Current user stack
	 adjspb 8		;Room for pushed long if needed
	 bicpsrw flag_s		;Back to supervisor stack
	else
	 addr	4+4+2+2(sp),r0	;Stack pointer before trap taken
	 adjspb	4+4+2+2+4+4+4	;Working space and room for pushed long
	 movmd	4+4+2+2+4+4+4(sp),(sp),3 ;Copy r0, PC, MOD, PSR to top of stack
	endif			;TOS holds original r0, r0 holds sp

	proc	1		;These labels are global
fpu_psr: blkw			;PSR flags for comparison updated on exit
fpu_savmod: blkw		;MOD of calling routine:w
fpu_op:	blkd			;Opcode pointer, updated on exit
fpu_savr0: blkd			;Saved r0 value, r0 holds sp on entry
fpu_efsr: 			;Error status returned here
	reg	[all]
fpu_sp: blkd			;SP of calling routine (U/S) updated on exit
fpu_savsp: blkd			;Original fpu_sp, not updated
fpu_mod: blkd			;MOD of calling routine:D (used as pointer)
fpu_sb: blkd			;SB of calling routine
fpu_imm: blkd	2		;Immediate operands put here and reversed
fpu_pc:	blkd			;Original PC of calling routine
fpu_tfsr: blkd			;Original FSR for error checks
	code

	mode	fp,fpu_tfsr
fpu_r0:	blkd			;CPU registers addressed with FP mode
fpu_r1:	blkd
fpu_r2:	blkd
fpu_r3:	blkd
fpu_r4:	blkd
fpu_r5:	blkd
fpu_r6:	blkd
fpu_r7:	blkd
	mode	pc

	movd	r0,fpu_sp	;SP parameter pointer
	movd	r0,fpu_savsp	;This is never altered
	movd	fpu_savr0,fpu_r0 ;This is the real r0
	movzwd	fpu_savmod,fpu_mod ;Need Dword MOD to use as pointer
	cond	cpu532
	 tbitb cfg_de,m_config
	 if	fs		;DE mode selected if FS
	  movd	10*tde_max+tde_psb(tde_vct),r0 ;SB pointer for UND trap
	  movd	(r0),fpu_sb	;Calling routine's SB
	 else
	cend
	  movd (fpu_mod),fpu_sb	;Calling routine's SB
	cond	cpu532
	 endif
	cend
	movd	fpu_op,fpu_pc	;Original PC

	movd	freg_fsr,fpu_tfsr ;Original FSR for error checks
	inssb	0,freg_fsr,0,3	;Clear trap status
	movqd	0,fpu_efsr	;Clear error status
	extsd	(fpu_op),r2,0,24 ;Complete opcode
	movd	r2,r5
	lshd	-8,r5		;MSword of opcode
	movd	r2,r0
	lshd	5-24,r0		;Gen 1
	extsd	r5,r1,14-8,5	;Gen 2
	extsd	r5,r3,10-8,4	;Opcode for mode 11
	tbitb	8,r2		;F/L bit for mode 11
	cmpb	h'be,r2		;Mode 11
	if	ne
	 cmpb	h'3e,r2		;Mode 9
	 if	eq
	  lshb	-1,r3		;Only bits 11-13 used for mode 9
	  sbitb	4,r3		;Mode 9 follows 11 in jump table
	 endif
	 tbitb	10,r2		;F/L bit for mode 9
	endif
	if	eq		;Must be mode 9 or mode 11
	 bsr	fp_sbrt:w	;FS if float, FC if long
	else
	 movqd	-1,fpu_efsr	;Non FPU opcode
	endif

	cmpqd	0,fpu_efsr	;Check for errors
	if	ne
	 movd	fpu_pc,fpu_op	;Restore original PC value before exit
	 movd	fpu_savsp,fpu_sp ;Restore this too
	endif

	tbitb	log flag_s,fpu_psr ;Check stack selection
	if	fs		;User stack if FS
	 bispsrw flag_s
	 lprd	sp,fpu_sp	;Restore user stack
	 bicpsrw flag_s
	else			;Supervisor stack, rebuild before exit
	 movd	fpu_sp,r0	;Stack position before trap
	 movmd	0(fp),-16(r0),4	;Stack may have moved, move needed data
	 addr	-16(r0),r1
	 lprd	fp,r1		;Stack frame may have moved
	endif

	pend	0		;Don't drop parameters

;TOS holds error status, PC, MOD, PSR for RETT

	cmpqd	0,tos
	blt	trp_fpu		;FPU trap if positive error status
	bgt	trp_und		;Continue with regular UND trap if negative
	cond	cpu532
	 lprw	mod,4(sp)
	 lprd	sb,10*tde_max+tde_ssb(tde_vct) ;Stored SB for UND trap
	cend
	rett			;Success, return to program


;Update error status, store in FPU_EFSR

fsr_err:
	save	[r0,r1,r2]
	movzbd	freg_fsr,r0	;Updated FSR
	extsb	r0,r2,0,3	;Get TT status
	movzbd	fpu_tfsr,r1	;Original version
	xorb	r0,r1		;See if UF or IF flags changed
	andb	r0,r1		;Only keep if set in updated version
	lshb	1,r0		;Enables to flag position
	andb	r0,r1		;Only if enable set
	andb	b'1010000,r1	;Mask all but UF and IF flags
	cmpqb	1,r2		;Underflow TT
	if	eq
	 tbitb	4,r1		;See if underflow trap requested
	else
	 cmpqb	6,r2		;Inexact result TT
	 if	eq
	  tbitb	6,r1		;See if inexact result trap requested
	 else
	  bispsrb flag_f	;Flag set for trap request
	 endif
	endif
	if	fs		;Don't merge TT if FC
	 orb	r2,r1		;Merge TT into returned FSR status
	endif
	movd	r1,fpu_efsr	;Return here
	restore	[r0,r1,r2]
	ret


;TOS:D holds shift count, R7 holds value
;Return adjusted R7, remaining bits left justified in TOS
;Inexact result set as needed

fpu_shf:
	lproc
cnt:	blkd
	reg	[r1,r2]
	code

	movd	cnt,r1
	cmpqd	0,r1
	if	le
	 lshd	r1,r7		;Shift mantissa
	 movqd	0,r2		;No rounding needed
	disp	2
	else
	disp	1
	 movd	r7,r2
	 cmpb	-32,r1
	 if	lt
	  ashd	r1,r7
	  addd	32,r1		;Number of upper bits kept
	  lshd	r1,r2		;R2 holds lower bits lost
	 else
	  cmpb	-64,r1
	  if	lt
	   addd	32,r1
	   ashd	r1,r2
	  else
	   ashd	-16,r2
	   orw	r7,r2		;Set lower bits for rounding status
	  endif
	  ashd	-31,r7		;Sign extend R7
	 endif
	 cmpqd	0,r2
	 if	ne		;1 bits shifted out of range if NE
	  sbitb	6,freg_fsr	;Inexact result
	  inssb	b'110,freg_fsr,0,3 ;Inexact result
	 endif
	endif
	movd	r2,cnt

	pend	0

;TOS:D holds shift count, R6/R7 holds value
;Return adjusted R6/R7, remaining bits left justified in TOS
;Inexact result set as needed

fpu_shl:
	lproc
cnt:	blkd
	reg	[r1,r2,r4]
	code

	movd	cnt,r1
	cmpqd	0,r1
	if	le
	 if	ne
	  cmpd	32,r1
	  if	le
	   movd	r6,r7
	   movqd 0,r6
	   subd	32,r1
	  endif
	  cmpqd	0,r1
	  if	ne
	   movd	r6,r2
	   lshd	r1,r6
	   lshd	r1,r7		;Shift mantissa
	   subd	32,r1
	   lshd	r1,r2		;Upper bits of R6
	   ord	r2,r7		;Merge into lower bits of R7
	  endif
	 endif
	 movqd	0,r2		;No rounding needed
	disp	2
	else
	disp	1
	 cmpd	-32,r1
	 if	lt
	  movd	r6,r2
	  movd	r7,r4
	  lshd	r1,r6
	  ashd	r1,r7
	  addd	32,r1		;Number of upper bits kept
	  lshd	r1,r2		;R2 holds lower bits lost
	  lshd	r1,r4		;Lower bits of R7
	  ord	r4,r6		;Merge into upper bits of R6
	 disp	2
	 else
	 disp	1
	  cmpd	-64,r1
	  if	lt
	   movd	r6,r2
	   movd	r7,r6
	   ashd -31,r7		;Sign extend R7
	   addd	32,r1
	   cmpqd 0,r1
	   if	ne
	    movd r6,r4
	    lshd r1,r2
	    ashd r1,r6
	    addd 32,r1		;Number of upper bits kept
	    lshd r1,r4		;Lower bits of R6
	    ord	r4,r2		;Merge into upper bits of R2
	   endif
	  else
	   cmpd	-96,r1
	   if	lt
	    movd r7,r2
	    ashd -31,r7
	    movd r7,r6
	    addd 64,r1
	    ashd r1,r2
	   else
	    movd r7,r2
	    ashd -16,r2
	    orw	r6,r2
	    orw	r7,r2
	    lshd -16,r6
	    orw	r6,r2
	    ashd -31,r7
	    movd r7,r6
	   endif
	  endif
	 endif
	 cmpqd	0,r2
	 if	ne		;1 bits shifted out of range if NE
	  sbitb 6,freg_fsr	;Inexact result
	  inssb	b'110,freg_fsr,0,3 ;Inexact result
	 endif
	endif
	movd	r2,cnt

	pend	0


;Shift R6,R7 according to TOS:D
;Negative shifts set inexact result as needed
;Positive shifts must be in range

fpu_sf:
	lproc
cnt:	blkd			;Shift count
	reg	[r1,r2]
	code

	movd	cnt,r1
	subd	r1,r6		;Adjust exponent
	movd	r1,tos
	bsr	fpu_shf
	movd	tos,r2
	cmpqd	0,r2		;Round according to FSR if not exact
	if	ne		;1 bits shifted out of range if NE
	 extsb	freg_fsr,r1,7,2	;Rounding mode
	 cmpqb	0,r1		;Round to nearest if 0, Round toward 0 if not
	 if	eq
	  cbitb 31,r2		;Rounding needed if FS
	  if	fs
	   cmpqd 0,r2		;See if exactly half way
	   if	ne		;More than half way if NE
	    addqd 1,r7		;Round up
	   else
	    tbitb 0,r7
	   orif fs		;Round to even
	   endif
	  endif
	 endif
	endif

	pend

;Shift R3,R6/R7 according to TOS:D
;Negative shifts set inexact result as needed
;Positive shifts must be in range

fpu_sl:
	lproc
cnt:	blkd			;Shift count
	reg	[r1,r2,r4]
	code

	movd	cnt,r1
	subd	r1,r3		;Adjust exponent
	movd	r1,tos
	bsr	fpu_shl
	movd	tos,r2
	cmpqd	0,r2
	if	ne		;1 bits shifted out of range if NE
	 extsb	freg_fsr,r1,7,2	;Rounding mode
	 cmpqb	0,r1		;Round to nearest if 0, Round toward 0 if not
	 if	eq
	  cbitb 31,r2		;Rounding needed if FS
	  if	fs
	   cmpqd 0,r2		;See if exactly half way
	   if	ne		;More than half way if NE
	    addqd 1,r6		;Round up
	    addcd 0,r7
	   else
	    tbitb 0,r6
	   orif fs		;Round to even
	   endif
	  endif
	 endif
	endif

	pend


;Normalize R6,R7
;Highest bit number in TOS:B (0-31)

fpu_nf:
	lproc
bit:	blkb
	reg	[r1]
	code

	cmpqd	0,r7
	if	eq
	 movqd	0,r6
	else
	 lmo	r7,r1		;Get left-most 1 in R1
	 subb	bit,r1		;Shift factor for R7
	 movxbd	r1,r1
	 negd	r1,tos		;Shift factor for R6,R7
	 bsr	fpu_sf		;Shift R6,R6 according to TOS
	 movzbd	bit,r1
	 addqd	1,r1
	 tbitb	r1,r7		;Rounding might have produced a carry
	 if	fs
	  movqd	-1,tos
	  bsr	fpu_sf		;Bit 0 is 0 if carry
	 endif
	endif
	pend

;Normalize R3, R6/R7
;Highest bit number in TOS:B (0-63)

fpu_nl:
	lproc
bit:	blkb
	reg	[r1]
	code

	movd	r7,r1
	ord	r6,r1
	cmpqd	0,r1
	if	eq
	 movqd	0,r3
	disp	2
	else
	disp	1
	 cmpqd	0,r7
	 if	eq
	  lmo	r6,r1
	 else
	  lmo	r7,r1		;Get left-most 1 in R1
	  addb	32,r1
	 endif
	 subb	bit,r1		;Shift factor for R7
	 movxbd	r1,r1
	 negd	r1,tos		;Shift factor for R6,R7
	 bsr	fpu_sl		;Shift R6,R6 according to TOS
	 movzbd	bit,r1
	 addqd	1,r1
	 cmpb	32,r1
	 if	hi
	  tbitb	r1,r6		;Rounding might have produced a carry
	 else
	  subb	32,r1
	  tbitb	r1,r7
	 endif
	 if	fs
	  movqd	-1,tos
	  bsr	fpu_sl		;Bit 0 is 0 if carry
	 endif
	endif
	pend


;R4,R5,R6,R7 hold exponents and mantissas
;Adjust R4,R5 to match exponent of R6/R7

fpu_e1f:
	subd	r6,r4		;Difference in exponents
	if	cc		;R6 is smaller if CC
	 negd	r4,tos
	 bsr	fpu_sf		;Shift for count on stack
	else			;R4 is negative
	 save	[r6,r7]
	 movd	r4,tos
	 addd	r4,r6		;Original R4 to R6
	 movd	r5,r7
	 bsr	fpu_sf		;Right shift for count on stack
	 movd	r7,r5
	 restore [r6,r7]
	endif
	movd	r6,r4
	ret

;Adjust R2,R4/R5 for exponent in R3

fpu_e1l:
	subd	r3,r2		;Difference in exponents
	if	cc		;R3 is smaller if CC
	 negd	r2,tos
	 bsr	fpu_sl		;Shift for count on stack
	else			;R4 is negative
	 save	[r3,r6,r7]
	 movd	r2,tos
	 addd	r2,r3		;Original R2 to R3
	 movd	r4,r6
	 movd	r5,r7
	 bsr	fpu_sl		;Right shift for count on stack
	 movd	r6,r4
	 movd	r7,r5
	 restore [r3,r6,r7]
	endif
	movd	r3,r2
	ret


;R6 holds exponent, R7 holds mantissa
;Put into (R1), sign bit must be clear

fpu_t1f:
	save	[r0]
	cmpqd	0,r7
	sgtb	r0		;R0=1 if negative, 0 if positive
	if	gt
	 negd	r7,r7		;Need absolute value
	endif
	movb	23,tos		;Bit position for highest bit
	bsr	fpu_nf
	cmpqd	0,r6		;Must be positive
	if	gt
	 movqd	0,r6
	 movqd	0,r7		;Everything is 0
	 sbitb	4,freg_fsr	;Underflow
	 inssb	1,freg_fsr,0,3	;Underflow
	else
	 cmpd	h'FE,R6		;Can't be more than this
	 if	lo
	  inssb	b'010,freg_fsr,0,3 ;Overflow
	 endif
	endif
	bsr	fsr_err		;Update error status in FPU_EFSR
	cmpqd	0,fpu_efsr	;Check for errors
	if	eq
	 cbitb	7,3(r1)		;Make positive for now
	 cmpqb	0,r0
	 if	ne
	  sbitb	7,3(r1)		;Make negative
	 endif
	 inssd	r7,(r1),0,23	;Install mantissa
	 inssb	r6,2(r1),23-16,8 ;Install exponent
	endif
	restore	[r0]
	ret

;R3 holds exponent, put R6/R7 into (R1)

fpu_t1l:
	save	[r0]
	cmpqd	0,r7		;Check sign
	sgtb	r0		;R0=1 if negative, 0 if positive
	if	gt
	 negq	r6,r7
	endif
	movb	52,tos		;Bit position for highest bit
	bsr	fpu_nl
	cmpqd	0,r3		;Must be positive
	if	gt
	 movqd	0,r3
	 movqd	0,r6
	 movqd	0,r7		;Everything is 0
	 sbitb	4,freg_fsr	;Underflow
	 inssb	1,freg_fsr,0,3	;Underflow status
	else
	 cmpd	h'7FE,R3	;Can't be more than this
	 if	lo
	  inssb	b'010,freg_fsr,0,3 ;Overflow status
	 endif
	endif
	bsr	fsr_err		;Update error status in FPU_EFSR
	cmpqd	0,fpu_efsr	;Check for errors
	if	eq
	 cbitb	7,7(r1)		;Make positive for now
	 cmpqb	0,r0
	 if	ne
	  sbitb	7,7(r1)		;Make negative
	 endif
	 movd	r6,(r1)		;Install mantissa LSDword
	 inssd	r7,4(r1),0,52-32 ;Install mantissa MSDword
	 inssw	r3,6(r1),52-48,11 ;Install exponent
	endif
	restore	[r0]
	ret

;R7 holds integer, R2 holds opcode
;Put into (R1)

fpu_t1i:
	tbitb	9,r2		;Check for Dword operand
	if	fs
	 movd	r7,tos
	 movd	r7,(r1)
	else
	 tbitb	8,r2		;Word if FS
	 if	fs
	  movxwd r7,tos
	  movw	r7,(r1)
	 else
	  movxbd r7,tos		;Byte only
	  movb	r7,(r1)
	 endif
	endif
	cmpd	tos,r7
	if	ne
	 inssb	b'010,freg_fsr,0,3 ;Overflow
	endif 
	br	fsr_err		;Update error status


;R3 holds opcode for FPU simulation, bit 4 set if mode 9
;R2 holds complete opcode, R0=gen1, R1=gen2
;FS if 32 bit float, FC if 64 bit long

fp_sbrt:
	casew	fp_sbr1[r3:w]
fp_sbr1:
	dw	fpu11_0-fp_sbrt	;ADDf
	dw	fpu11_1-fp_sbrt	;MOVf
	dw	fpu11_2-fp_sbrt	;CMPf
	dw	fpu11_3-fp_sbrt
	dw	fpu11_4-fp_sbrt	;SUBf
	dw	fpu11_5-fp_sbrt	;NEGf
	dw	fpu11_6-fp_sbrt
	dw	fpu11_7-fp_sbrt
	dw	fpu11_8-fp_sbrt	;DIVf
	dw	fpu11_9-fp_sbrt
	dw	fpu11_a-fp_sbrt
	dw	fpu11_b-fp_sbrt
	dw	fpu11_c-fp_sbrt	;MULf
	dw	fpu11_d-fp_sbrt	;ABSf
	dw	fpu11_e-fp_sbrt
	dw	fpu11_f-fp_sbrt

	dw	fpu9_0-fp_sbrt	;MOVif
	dw	fpu9_1-fp_sbrt	;LFSR
	dw	fpu9_2-fp_sbrt	;MOVLF
	dw	fpu9_3-fp_sbrt	;MOVFL
	dw	fpu9_4-fp_sbrt	;ROUNDfi
	dw	fpu9_5-fp_sbrt	;TRUNCfi
	dw	fpu9_6-fp_sbrt	;SFSR
	dw	fpu9_7-fp_sbrt	;FLOORfi


;FPU_OP=> displacement, Add to R0 and advance pointer

fpu_disp:
	save	[r1]
	movd	(fpu_op),r1
	tbitb	7,r1		;Check for :B
	if	fc
	 lshb	1,r1
	 ashb	-1,r1
	 movxbd	r1,r1
	 addqd	1,fpu_op
	else
	 rotw	8,r1		;Swap lower 2 bytes
	 tbitb	14,r1		;Check for :W
	 lshw	2,r1
	 ashw	-2,r1
	 if	fc
	  movxwd r1,r1
	  addqd	2,fpu_op
	 else
	  rotd	16,r1		;Swap upper and lower words
	  rotw	8,r1		;Swap lower word
	  addqd	4,fpu_op
	 endif
	endif
	addd	r1,r0
	restore	[r1]
	ret

;------------------------
;Addressing mode routines
;------------------------

;R0 holds gen, get pointer to operand in R0
;R1 holds 0 for gen1 or 1 for gen2
;R2 holds complete opcode

fpu_g1:
	casew	fpu_g1t:b[r0:w]
fpu_g1t:
	dw	fpu_g00-fpu_g1	;FPU Register 0
	dw	fpu_g01-fpu_g1	;FPU Register 1
	dw	fpu_g02-fpu_g1	;FPU Register 2
	dw	fpu_g03-fpu_g1	;FPU Register 3
	dw	fpu_g04-fpu_g1	;FPU Register 4
	dw	fpu_g05-fpu_g1	;FPU Register 5
	dw	fpu_g06-fpu_g1	;FPU Register 6
	dw	fpu_g07-fpu_g1	;FPU Register 7

	dw	fpu_g08-fpu_g1	;Register 0 indirect
	dw	fpu_g09-fpu_g1	;Register 1 indirect
	dw	fpu_g0a-fpu_g1	;Register 2 indirect
	dw	fpu_g0b-fpu_g1	;Register 3 indirect
	dw	fpu_g0c-fpu_g1	;Register 4 indirect
	dw	fpu_g0d-fpu_g1	;Register 5 indirect
	dw	fpu_g0e-fpu_g1	;Register 6 indirect
	dw	fpu_g0f-fpu_g1	;Register 7 indirect

	dw	fpu_g10-fpu_g1	;FP relative indirect
	dw	fpu_g11-fpu_g1	;SP relative indirect
	dw	fpu_g12-fpu_g1	;SB relative indirect
	dw	fpu_g13-fpu_g1	;Undefined
	dw	fpu_g14-fpu_g1	;Immediate
	dw	fpu_g15-fpu_g1	;Absolute (displacement)
	dw	fpu_g16-fpu_g1	;External
	dw	fpu_g17-fpu_g1	;TOS

	dw	fpu_g18-fpu_g1	;FP memory space
	dw	fpu_g19-fpu_g1	;SP memory space
	dw	fpu_g1a-fpu_g1	;SB memory space
	dw	fpu_g1b-fpu_g1	;PC memory space
	dw	fpu_g1c-fpu_g1	;Indexed:B
	dw	fpu_g1d-fpu_g1	;Indexed:W
	dw	fpu_g1e-fpu_g1	;Indexed:D
	dw	fpu_g1f-fpu_g1	;Indexed:Q

fpu_g00:			;Register 0 (CPU or FPU)
fpu_g01:			;Register 1 (CPU or FPU)
fpu_g02:			;Register 2 (CPU or FPU)
fpu_g03:			;Register 3 (CPU or FPU)
fpu_g04:			;Register 4 (CPU or FPU)
fpu_g05:			;Register 5 (CPU or FPU)
fpu_g06:			;Register 6 (CPU or FPU)
fpu_g07:			;Register 7 (CPU or FPU)
	cmpb	h'3e,r2		;Check for mode 9
	tbitb	8,r2		;FS if float and not mode 9
	if	ne		;Always 2 floats of same size if not 9
	 if	fc		;Long if FC
	  tbitb 0,r0		;See if even
	  if	fs		;Odd if FS
	   addqd 7,r0		;1/3/5/7=>8/10/12/14
	  endif
	 endif
	 addr	freg_f0[r0:d],r0
	 cmpqb	1,r1		;Destination is FPU register if gen2
	 if	eq
	  sbitb	16,freg_fsr	;Set Register Modified Bit
	 endif
	disp	2
	else			;Mode 9
	disp	1
	 save	[r3]
	 movzbd	11,r3
	 extd	r3,r2,r3,3	;Op index
	 tbitb	10,r2		;FS if float, FC if long
	 cmpqb	4,r3		;ROUNDfi, TRUNCfi, FLOORfi, SFSR
	 if	ls
	  cmpqb	0,r1		;FPU register if gen1
	 else
	  cmpqb	0,r3		;MOVif
	  if	eq
	   cmpqb 1,r1		;Gen 2 is FPU register
	  else
	   cmpqb 2,r3		;MOVLF
	   if	eq		;Bit 10 is set if EQ
	    cmpqb 1,r1		;Gen1 is long, gen2 is float
	    if	ne
	     bicpsrb flag_f	;FC for gen1 (long)
	     bispsrb flag_z	;Make EQ
	    endif
	   else
	    cmpqb 3,r3		;MOVFL
	    if	eq		;Bit 10 is clear if EQ
	     cmpqb 1,r1		;Gen2 is long, gen1 is float
	     if	ne
	      bispsrb flag_f	;FS for gen1 (float)
	      bispsrb flag_z	;Make EQ
	     endif
	    endif
	   endif
	  endif
	 endif
	 restore [r3]
	disp	2
	orif	eq		;FPU register if EQ
	disp	1
	 negd	r0,r0		;CPU register
	 addr	fpu_r0[r0:d],r0
	endif
	ret

fpu_g08:			;CPU register 0 indirect
fpu_g09:			;CPU register 1 indirect
fpu_g0a:			;CPU register 2 indirect
fpu_g0b:			;CPU register 3 indirect
fpu_g0c:			;CPU register 4 indirect
fpu_g0d:			;CPU register 5 indirect
fpu_g0e:			;CPU register 6 indirect
fpu_g0f:			;CPU register 7 indirect
	andb	b'111,r0	;Mask all but register number
	negd	r0,r0		;Negate index
	movd	fpu_r0[r0:d],r0
	br	fpu_disp	;Add displacement to R0

fpu_g10:			;FP memory relative (indirect)
	movd	0(fp),r0	;Original FP
	bsr	fpu_disp	;Pointer to pointer
	movd	(r0),r0		;Pointer in R0
	br	fpu_disp	;Add disp. 2
	
fpu_g11:			;SP memory relative (indirect)
	movd	fpu_sp,r0	;Original SP
	bsr	fpu_disp	;Pointer to pointer
	movd	(r0),r0		;Pointer in R0
	br	fpu_disp	;Add disp. 2
	
fpu_g12:			;SB memory relative (indirect)
	movd	fpu_sb,r0	;Original SB
	bsr	fpu_disp	;Pointer to pointer
	movd	(r0),r0		;Pointer in R0
	br	fpu_disp	;Add disp. 2
	
fpu_g13:			;Undefined
	inssb	b'101,freg_fsr,0,3 ;Invalid operation
	ret

fpu_g14:			;Immediate
	save	[r3]
	movzbd	11,r3
	extd	r3,r2,r3,3	;Op index for mode 9
	cmpb	h'3e,r2		;Check for mode 9
	tbitb	8,r2		;FS if float and not mode 9
	if	ne		;Always 2 floats of same size if not 9
	 movd	(fpu_op),r0	;MSDword
	 rotw	8,r0
	 rotd	16,r0
	 rotw	8,r0		;Byte order reversed
	 if	fc		;Long if FC
	  movd	r0,4+fpu_imm
	  movd	4(fpu_op),r0	;LSDword
	  rotw	8,r0
	  rotd	16,r0
	  rotw	8,r0		;Byte order reversed
	  movd	r0,fpu_imm
	  addd	8,fpu_op
	 else
	  movd	r0,fpu_imm
	  addqd	4,fpu_op
	 endif
	disp	2
	else			;Mode 9
	disp	1
	 tbitb	10,r2		;FS if float, FC if long
	 cmpqb	4,r3		;ROUNDfi, TRUNCfi, FLOORfi, SFSR
	disp	2
	orif	ls		;Immediate only for gen1
	disp	1
	 cmpqb	2,r3		;MOVLF
	 if	eq		;Bit 10 is set if EQ
	  bicpsrb flag_f	;FC for gen1 (long)
	 else
	  cmpqb 3,r3		;MOVFL, bit 10 is clear if EQ
	  if	eq
	   bispsrb flag_f	;FS for gen1 (float)
	  endif
	 endif
	disp	2
	orif	eq
	disp	1
	 movd	(fpu_op),fpu_imm ;Copy parameter
	 movzbd	8,r0
	 extd	r0,r2,r0,2	;Size of immediate operand
	 addqd	1,r0		;Number of bytes
	 addd	r0,fpu_op
	 cmpqb	2,r0
	 if	eq
	  rotw	8,fpu_imm
	 else
	  cmpqb	4,r0
	  if	eq
	   movd	fpu_imm,r0
	   rotw	8,r0
	   rotd	16,r0
	   rotw	8,r0
	   movd	r0,fpu_imm
	  endif
	 endif
	endif
	addr	fpu_imm,r0	;Point to properly byte ordered operand
	restore	[r3]
	ret

fpu_g15:			;Absolute
	movqd	0,r0		;Displacement is all there is
	br	fpu_disp	;Pointer to address

fpu_g16:			;External
	movqd	0,r0
	bsr	fpu_disp	;Link table entry
	movd	4(fpu_mod)[r0:d],r0 ;Pointer to value
	br	fpu_disp	;Add 2nd offset

fpu_g17:			;TOS
	save	[r5]		;Put size of operand in R5
	cmpb	h'3e,r2		;Check for mode 9
	tbitb	8,r2		;FS if float and not mode 9
	if	ne		;Always 2 floats of same size if not 9
	 if	fc		;Long if FC
	  movzbd 8,r5		;Size of operand
	 else
	  movqd	4,r5
	 endif
	disp	2
	else			;Mode 9
	disp	1
	 movzbd	11,r5
	 extd	r5,r2,r5,3	;Op index for mode 9
	 tbitb	10,r2		;FS if float, FC if long
	 cmpqb	4,r5		;ROUNDfi, TRUNCfi, FLOORfi, SFSR
	 if	ls
	  cmpqb	0,r1		;Float/long if gen1
	 else
	  cmpqb	0,r5		;MOVif
	  if	eq
	   cmpqb 1,r1		;Gen 2 is float/long
	  else
	   cmpqb 2,r5		;MOVLF
	   if	eq		;Bit 10 is set if EQ
	    cmpqb 1,r1		;Gen1 is long, gen2 is float
	    if	ne
	     bicpsrb flag_f	;FC for gen1 (long)
	     bispsrb flag_z	;Make EQ
	    endif
	   else
	    cmpqb 3,r5		;MOVFL
	    if	eq		;Bit 10 is clear if EQ
	     cmpqb 1,r1		;Gen2 is long, gen1 is float
	     if	ne
	      bispsrb flag_f	;FS for gen1 (float)
	      bispsrb flag_z	;Make EQ
	     endif
	    endif
	   endif
	  endif
	 endif
	disp	2
	orif	eq		;Float/long if EQ
	disp	1
	 movzbd	8,r5
	 extd	r5,r2,r5,2	;Size of immediate operand
	 addqd	1,r5		;Number of bytes
	endif
	cmpqb	0,r1		;Check for gen1
	if	eq		;POP
	 movd	fpu_sp,r0	;Address of operand
	 addd	r5,fpu_sp
	else
	 subd	r5,fpu_sp	;PUSH
	 movd	fpu_sp,r0	;Address of operand
	endif
	restore [r5]
	ret

fpu_g18:			;FP relative
	movd	0(fp),r0	;Original FP
	br	fpu_disp

fpu_g19:			;SP relative
	movd	fpu_sp,r0	;Original SP
	br	fpu_disp

fpu_g1a:			;SB relative
	movd	fpu_sb,r0	;Original SB
	br	fpu_disp

fpu_g1b:			;PC relative
	movd	fpu_pc,r0	;Original PC
	br	fpu_disp

;Indexed mode
;Index byte: bits 0-2 are register, bits 3-7 are real gen mode

fpu_g1c:			;Index:B
fpu_g1d:			;Index:W
fpu_g1e:			;Index:D
fpu_g1f:			;Index:Q
	save	[r4,r5,r6]
	cmpqb	0,r1		;Check for gen1
	if	eq
	 movb	(fpu_op),r4	;Index byte
	 addqd	1,fpu_op	;Skip over index byte
	 extsb	1(fpu_pc),r5,14-8,5 ;Gen2
	 cmpb	h'1c,r5		;See if this is also indexed
	 if	ls		;Indexed if LS
	  addqd	1,fpu_op	;Skip 2nd index byte
	 endif
	else
	 extsb	2(fpu_pc),r5,19-16,5 ;Gen1
	 cmpb	h'1c,r5		;See if this is also indexed
	 if	ls		;Indexed if LS
	  movb	4(fpu_pc),r4	;Index byte if both indexed
	 else
	  movb	3(fpu_pc),r4	;Index byte if only gen2 indexed
	 endif
	endif

	extsd	r0,r6,0,2	;Index size
	movqd	1,r5
	lshd	r6,r5		;Size in bytes of index multiplier
	extsd	r4,r6,0,3	;Index register
	negd	r6,r6
	movd	fpu_r0[r6:d],r6	;Value of index register
	muld	r5,r6		;Actual index
	extsd	r4,r0,3,5	;Real gen mode

	cmpqb	7,r0		;R0-R7 is a special case
	if	hs
	 negd	r0,r0
	 movd	fpu_r0[r0:d],r0	;Gen mode is (Rn) but no offset
	else
	 bsr	fpu_g1		;Process real gen mode now
	endif
	addd	r6,r0
	restore	[r4,r5,r6]
	ret


;-----------------------------
;General manipulation routines
;-----------------------------

;R0 holds source gen1, R1 holds target gen2
;R2 holds complete opcode
;FPU_OP=>start of opcode, advanced on exit
;Get pointer to source operand in R0
;Pointer to target operand in R1

fpu_g2:
	addqd	3,fpu_op	;Point to index bytes or displacements
	cmpb	h'1c,r1		;See if gen2 is indexed
	if	ls
	 cmpb	h'1c,r0		;If only gen2 indexed then increment pointer
	 if	hi
	  addqd	1,fpu_op	;Displacements start after index byte
	 endif
	endif
	save	[r1]
	movqd	0,r1		;Gen1
	bsr	fpu_g1		;Get source operand pointer
	restore	[r1]
	save	[r0]
	movd	r1,r0		;Gen2
	movqd	1,r1		;Gen2
	bsr	fpu_g1		;Get target operand pointer
	movd	r0,r1		;Move to R1
	restore	[r0]		;Source pointer restored
	ret

;R0 holds gen1, R1 holds gen2, R2 holds opcode
;Return source float as integer in R7, remaining bits in R6
;Destination pointer in R1

fpu_p2i:
	bsr	fpu_g2		;Get pointers in R0,R1
	tbitb	10,r2		;FS if float
	if	fs
	 extsd	(r0),r7,0,23	;Mantissa
	 extsd	2(r0),r6,23-16,8 ;Exponent
	 cmpqd	0,r6
	 if	ne
	  sbitb	23,r7		;This is always set
	  subd	h'7f+23,r6	;Shift factor
	  cmpd	30-23,r6
	  if	lt		;Too big if LT
	   inssb b'010,freg_fsr,0,3 ;Overflow condition
	  else
	   movd	r6,tos		;Shift factor
	   bsr	fpu_shf		;Shift R7 according to TOS
	   movd	tos,r6		;Remainder left justified
	   tbitb 7,3(r0)	;Check sign
	  endif
	 endif
	else 
	 movd	(r0),r6		;LSDword
	 extsd	4(r0),r7,0,52-32 ;Mantissa
	 extsd	6(r0),r3,52-48,11 ;Exponent
	 cmpqd	0,r3
	 if	ne
	  sbitb	52-32,r7	;This is always set
	  subd	h'3ff+52,r3	;Shift factor
	  cmpd	30-52,r3
	  if	lt		;Too big if LS
	   inssb b'010,freg_fsr,0,3 ;Overflow condition
	  else
	   movd	r3,tos		;Shift factor
	   bsr	fpu_shl		;Shift R7 according to TOS
	   movd	r6,r7		;Mantissa
	   movd	tos,r6		;Remainder left justified
	   tbitb 7,7(r0)	;Check sign
	  endif
	 endif
	endif
	if	fs
	 negd	r7,r7
	endif
	ret


;R0 holds gen1, R1 holds gen2; convert to pointers
;Get source exp in R4, mantissa in R5
;Get target exp in R6, mantissa in R7
;Sign bit of target register is cleared

fpu_p2f:
	bsr	fpu_g2		;Convert gens to pointers
	extsd	2(r0),r4,23-16,8 ;Source exponent
	extsd	(r0),r5,0,23	;Source mantissa
	cmpqb	0,r4		;Check for 0
	if	ne
	 sbitb	23,r5		;This is always 1
	endif
	tbitb	7,3(r0)		;Check sign
	if	fs
	 negd	r5,r5
	endif
	extsd	2(r1),r6,23-16,8 ;Target exponent
	extsd	(r1),r7,0,23	;Target mantissa
	cmpqb	0,r6		;Check for 0
	if	ne
	 sbitb	23,r7
	endif
	tbitb	7,3(r1)		;Check sign and clear for now
	if	fs
	 negd	r7,r7
	endif
	ret

;R0 gen1, R1 holds gen2; convert to pointers
;Get source exponent in R2, mantissa in R4/R5
;Sign bit of target register is cleared
;Get target exponent in R3, mantissa in R6/R7

fpu_p2l:
	bsr	fpu_g2		;Convert gens to pointers
	extsd	6(r0),r2,52-48,11 ;Source exponent
	movd	(r0),r4		;Source mantissa LSDword
	extsd	4(r0),r5,0,52-32 ;Source mantissa MSDword
	cmpqd	0,r2		;Check for 0
	if	ne
	 sbitb	52-32,r5	;This is always 1
	endif
	tbitb	7,7(r0)		;Check sign
	if	fs
	 negq	r4,r5
	endif	 
	extsd	6(r1),r3,52-48,11 ;Target exponent
	movd	(r1),r6		;Target mantissa LSDword
	extsd	4(r1),r7,0,52-32 ;Target mantissa MSDword
	cmpqd	0,r3		;Check for 0
	if	ne
	 sbitb	52-32,r7	;This is always 1
	endif
	tbitb	7,7(r1)		;Check sign
	if	fs
	 negq	r6,r7
	endif	 
	ret


;---------------------------------
;Actual opcode simulation routines
;---------------------------------

fpu11_3:
fpu11_6:
fpu11_7:
fpu11_9:
fpu11_a:
fpu11_b:
fpu11_e:
fpu11_f:
	inssb	b'100,freg_fsr,0,3 ;Illegal instruction
	ret			;These don't exist

fpu11_0:			;ADDf
	if	fs
	 bsr	fpu_p2f		;Get 2 32 bit float parameters
	 bsr	fpu_e1f		;Adjust exponent of source parameter
	 addd	r5,r7
	 bsr	fpu_t1f		;Put R6,R7 into target in (R1)
	else
	 bsr	fpu_p2l		;Get 2 64 bit long parameters
	 bsr	fpu_e1l		;Adjust exponent of source paremeter
	 addd	r4,r6
	 addcd	r5,r7
	 bsr	fpu_t1l		;Put R3,R6/R7 into target in (R1)
	endif
	ret

fpu11_1:			;MOVf
	bsr	fpu_g2		;Get pointers to operands
	tbitb	8,r2		;FS if float
	if	fs
	 movd	(r0),(r1)
	else
	 movmd	(r0),(r1),2
	endif
	ret

fpu11_2:			;CMPf
	if	fs
	 bsr	fpu_p2f
	 movd	r5,r3
	 xord	r7,r3
	 tbitb	31,r3		;Check signs
	 if	fs
	  cmpd	r5,r7		;Compare signed mantissas
	 else
	  cmpb	r4,r6		;Compare exponents
	 orif	eq
	 endif
	else
	 bsr	fpu_p2l
	 movd	r5,r0
	 xord	r7,r0
	 tbitb	31,r0		;Check signs
	 if	fs
	  cmpd	r5,r7		;Signed mantissa MSDwords
	  if	eq
	   cmpd	r4,r6		;Mantissa LSDwords
	  endif
	 else
	  cmpw	r2,r3		;Compare exponents
	 orif	eq
	 endif
	endif
	bicpsrb	flag_l		;This is always 0 for FPU compares
	sprb	upsr,r2		;These must be retuned
	bicb	flag_l+flag_z+flag_n,fpu_psr ;These are updated
	andb	flag_l+flag_z+flag_n,r2 ;Keep only these
	orb	r2,fpu_psr	;CPU flags for return now correct
	ret

fpu11_4:			;SUBf
	if	fs
	 bsr	fpu_p2f		;Get 2 32 bit float parameters
	 bsr	fpu_e1f		;Adjust exponent of source parameter
	 subd	r5,r7
	 bsr	fpu_t1f		;Put R6,R7 into (R1)
	else
	 bsr	fpu_p2l		;Get 2 64 bit long parameters
	 bsr	fpu_e1l		;Adjust exponent of source paremeter
	 subd	r4,r6
	 subcd	r5,r7
	 bsr	fpu_t1l		;Put R3,R6/R7 into target register
	endif
	ret

fpu11_5:			;NEGf
	bsr	fpu11_1		;Move
	tbitb	8,r2		;Float if FS, long if FC
	if	fs
	 ibitb	7,3(r1)
	else
	 ibitb	7,7(r1)
	endif
	ret

fpu11_8:			;DIVf
	disp	2
	if	fs
	disp	1
	 bsr	fpu_p2f		;Get 2 32 bit float parameters
	 cmpqd	0,r5		;Can't divide by 0
	 if	eq
	  cmpqd	0,r7		;0/0 is invalid operation
	  if	eq
	   inssb b'101,freg_fsr,0,3 ;Invalid operation
	  else
	   inssb b'011,freg_fsr,0,3 ;Divide by 0
	  endif
	 disp	2
	 else
	 disp	1
	  subd	r4,r6		;Subtract exponents
	  addd	h'7f-32+23,r6	;Add offset bias
	  movd	r6,r3		;Exponent for R3,R6/R7
	  movd	r5,r2
	  xord	r7,r2		;Bit 31 is sign of quotient
	  absd	r5,r5
	  absd	r7,r6
	  movqd	0,r7		;Dividend in R6/R7
	  deid	r5,r6		;Divide mantissas
	  cmpqd	0,r7
	  if	eq
	   subd	32,r3		;Adjust exponent for another round
	   movd	r6,r7
	   movqd 0,r6
	   deid	r5,r6		;More bits for more precision
	  endif
	  movd	r7,tos
	  movd	r6,r7
	  movqd	0,r6
	  deid	r5,r6
	  movd	r7,r6
	  movd	tos,r7
	  cmpqd	0,r7
	  if	gt		;Sign must be positive
	   movd	r7,r6
	   movqd 0,r7
	   addd	32,r3
	  endif
	  movb	23,tos		;Highest bit
	  bsr	fpu_nl		;Normalize R3,R6/R7 for float in R3,R6
	  movd	r6,r7
	  cmpqd	0,r2
	  if	gt
	   negd	r7,r7
	  endif
	  movd	r3,r6		;Exp in R3, mantissa in R6
	 endif
	 bsr	fpu_t1f		;Put R6,R7 into (R1)
	disp	2
	else
	disp	1
	 bsr	fpu_p2l		;Get 2 64 bit long parameters
	 movd	r5,tos
	 ord	r4,tos
	 cmpqd	0,tos		;Can't divide by 0
	 if	eq
	  movd	r7,tos
	  ord	r6,tos
	  cmpqd	0,tos
	  if	eq
	   inssb b'101,freg_fsr,0,3 ;Invalid operation
	  else
	   inssb b'011,freg_fsr,0,3 ;Divide by 0
	  endif
	 disp	2
	 else
	 disp	1
	  subd	r2,r3		;Subtract exponents
	  addd	h'3ff+52,r3	;Add offset
	  save	[r0,r1,r2,r3]
	  movd	r5,tos
	  xord	r7,(sp)		;Sign of quotient in bit 31 of (SP)
	  cmpqd	0,r5
	  if	gt
	   negq	r4,r5
	  endif
	  cmpqd	0,r7
	  if	gt
	   negq	r6,r7
	  endif
	  movb	64,tos		;Exponent of quotient
	  movqd	0,r0
	  movqd	0,r1		;Quotient in R0/R1
	  movqd	0,r2
	  movqd	0,r3		;Dividend shifted through R2/R3
	  until	fs		;Go until bit 52 set of R0/R1 or dividend=0
	   save	[r2]
	   ord	r3,r2
	   ord	r6,r2
	   ord	r7,r2
	   cmpqd 0,r2
	   restore [r2]
	  quit	eq		;Done if dividend exhausted
	   addd	r0,r0
	   addcd r1,r1		;Shift quotient right
	   addqb -1,(sp)		;Decrement exponent
	   addd	r6,r6
	   addcd r7,r7
	   addcd r2,r2
	   addcd r3,r3		;Shift dividend
	   save	[r2,r3]
	   subd	r4,r2
	   subcd r5,r3		;Trial subtraction
	   if	cc
	    sbitb 0,r0
	    adjspb -8
	   else
	    restore [r2,r3]
	   endif
	   tbitb 54-32,r1	;2 extra bits for rounding
	  disp	2
	  endu
	  disp	1
	  movd	r0,r6
	  movd	r1,r7		;Quotient to R6/R7
	  movxbd tos,r4		;Exponent for quotient
	  movd	tos,r5		;Sign of quotient in bit 31
	  restore [r0,r1,r2,r3]
	  addd	r4,r3
	  cmpqd	0,r5
	  if	gt
	   negq	r6,r7
	  endif
	 endif
	 bsr	fpu_t1l		;Put R3,R6/R7 into target register
	endif
	ret

fpu11_c:			;MULf
	if	fs
	 bsr	fpu_p2f		;Get 2 32 bit float parameters
	 movd	r5,r2
	 xord	r7,r2		;Sign of product
	 absd	r5,r5		;Must be positive here
	 absd	r7,r7		;Must be positive here
	 movd	r6,r3		;Target exponent
	 addd	r4,r3		;Add exponents
	 subd	h'7f+23,r3	;Subtract bias offset
	 movd	r7,r6
	 movqd	0,r7		;Target now in R6/R7
	 meid	r5,r6
	 movb	23,tos
	 bsr	fpu_nl		;Normalize R3,R6/R7 to float in R3,R6
	 movd	r6,r7
	 movd	r3,r6
	 cmpqd	0,r2		;See if product should be negative
	 if	gt
	  negd	r7,r7
	 endif
	 bsr	fpu_t1f		;Put R6,R7 into (R1)
	disp	2
	else
	disp	1
	 bsr	fpu_p2l		;Get 2 64 bit long parameters
	 addd	r2,r3		;Add exponents
	 subd	h'3ff+52,r3	;Subtract bias offset
	 
	 save	[r0,r1,r2,r3]
	 movd	r5,r0
	 xord	r7,r0		;Sign of product
	 movd	r0,tos		;Save it
	 
	 cmpqd	0,r5		;Unsigned multiply coming up
	 if	gt
	  negq	r4,r5
	 endif
	 cmpqd	0,r7
	 if	gt
	  negq	r6,r7
	 endif
	 
	 movd	r4,r0
	 movqd	0,r1
	 meid	r6,r0		;LSDword*LSDword
	 
	 movd	r5,r2
	 movqd	0,r3
	 meid	r6,r2		;LSDword*MSDword
	 addd	r2,r1
	 addcd	0,r3
	 movd	r3,r2

	 movd	r2,tos
	 movd	r4,r2
	 movqd	0,r3
	 meid	r7,r2		;MSDword*LSDword
	 addd	r2,r1
	 addcd	tos,r3
	 movd	r3,r2
	 scsd	r3
	
	 movd	r3,tos
	 movd	r2,tos
	 movd	r5,r2
	 movqd	0,r3
	 meid	r7,r2		;MSDword*MSDword
	 addd	tos,r2
	 addcd	tos,r3		;Complete product now in R0/R1/R2/R3
	 
	 movzbd	64,r4		;Exponent of product
	 lmo	r3,r0
	 negd	r0,r0
	 addd	54-32,r0	;Shift factor, counting 2 extra rounding bits
	 subd	r0,r4		;Adjust exponent
	 movd	r3,r7
	 movd	r2,r6
	 lshd	r0,r7
	 lshd	r0,r6
	 cmpqd	0,r0
	 if	gt		;Right shift if GT
	  addd	32,r0
	  lshd	r0,r3		;Dropped bits of MSDword
	  ord	r3,r6		;Merge into upper bits of LSDword
	 else
	  if	ne
	   subd 32,r0
	   lshd r0,r2		;Dropped bits
	   ord	r2,r7		;Merge into lower bits of MSDword
	   lshd r0,r1		;Lowest bits
	   ord	r1,r6		;Lower bits of LSDword
	  endif
	 endif
	 movd	tos,r0		;Sign of product
	 cmpqd	0,r0		;See if product should be negative
	 if	gt
	  negq	r6,r7
	 endif
	 restore [r0,r1,r2,r3]
	 addd	r4,r3		;Adjustment to exponent
	 bsr	fpu_t1l		;Put R3,R6/R7 into target register
	endif
	ret

fpu11_d:			;ABSf
	bsr	fpu11_1		;Move
	tbitb	8,r2		;Float if FS, long if FC
	if	fs
	 cbitb	7,3(r1)
	else
	 cbitb	7,7(r1)
	endif
	ret

fpu9_0:				;MOVif
	bsr	fpu_g2		;Pointer to integer in R0
	tbitb	9,r2		;Check for Dword
	if	fs
	 movd	(r0),r7
	else
	 tbitb	8,r2		;Byte or word
	 if	fs		;Word if FS
	  movxwd (r0),r7
	 else
	  movxbd (r0),r7
	 endif
	endif
	
	tbitb	10,r2		;FS if float
	if	fs
	 movzbd	h'7f+23,r6	;Add offset bias to exponent
	 bsr	fpu_t1f		;Put R6,R7 into (R1)
	else
	 movd	r6,r3		;Exponent
	 movd	r7,r6
	 ashd	-31,r7		;Sign extend R7
	 movzwd	h'3ff+52,r3
	 bsr	fpu_t1l		;Put R6,R7 into (R1)
	endif
	ret

fpu9_1:				;LFSR
	bsr	fpu_g2		;R0=>value to load with
	movd	(r0),freg_fsr
	ret

fpu9_2:				;MOVLF
	bsr	fpu_g2		;Get pointers in R0,R1
	extsd	3(r0),r7,(51-27)-24,28 ;Upper 28 bits of Mantissa for rounding
	extsd	6(r0),r6,52-48,11 ;Exponent
	cmpqd	0,r6
	if	ne
	 addd	23-28+h'7F-H'3ff,r6 ;Convert, 5 extra bits added for rounding
	 sbitb	28,r7		;This is always set
	endif
	tbitb	7,7(r0)		;Check sign
	if	fs
	 negd	r7,r7
	endif
	bsr	fpu_t1f		;Install converted long
	ret

fpu9_3:				;MOVFL
	bsr	fpu_g2		;Get pointers in R0,R1
	extsd	(r0),r6,0,23	;Mantissa
	extsd	2(r0),r3,23-16,8 ;Exponent
	cmpqd	0,r3
	if	ne
	 sbitb	23,r6		;This is always set
	 addd	h'3FF-H'7f+(52-23),r3 ;Convert exponent
	endif
	movqd	0,r7
	tbitb	7,3(r0)		;Check sign
	if	fs
	 negd	r6,r6
	 movqd	-1,r7
	endif
	bsr	fpu_t1l		;Install converted float
	ret

fpu9_4:				;ROUNDfi
	bsr	fpu_p2i		;Get integer in R7, dest pointer in R1

	cbitb	31,r6		;Round according to R6
	if	fs
	 cmpqd 0,r6
	 if	ne		;More than halfway if NE
	  movd	r7,r5
	  ashd	-31,r5		;Sign extend R7 to R5
	  orb	1,r5		;0=>1, -1=>-1
	  addd	r5,r7
	  if	fs		;Check for rounding overflow
	   inssb b'010,freg_fsr,0,3 ;Overflow
	  endif
	 else			;Exactly halfway
	  tbitb	0,r7
	 orif	fs
	 endif
	endif

	br	fpu_t1i

fpu9_5:				;TRUNCfi
	bsr	fpu_p2i		;Get integer in R7, dest pointer in R1
	br	fpu_t1i

fpu9_6:				;SFSR
	bsr	fpu_g2		;R1=>destination address
	inssb	fpu_tfsr,freg_fsr,0,3 ;Original FSR TT status
	movd	freg_fsr,(r1)
	ret

fpu9_7:				;FLOORfi
	bsr	fpu_p2i		;Get integer in R7, dest pointer in R1

	cmpqd	0,r6
	if	ne
	 movd	r7,r5
	 ashd	-31,r5		;Sign extend R7 into R5
	 addd	r5,r7		;Add 0 if positive, -1 if negative
	 if	fs		;Check for rounding overflow
	  inssb b'010,freg_fsr,0,3 ;Overflow
	 endif
	endif

	br	fpu_t1i

;End of FPU32
